/************************************************************************
* Author    : Tiago Dionizio (tngd@mega.ist.utl.pt)                     *
* Library   : lzlib - Lua 5 interface to access zlib library functions  *
*                                                                       *
* Permission is hereby granted, free of charge, to any person obtaining *
* a copy of this software and associated documentation files (the       *
* "Software"), to deal in the Software without restriction, including   *
* without limitation the rights to use, copy, modify, merge, publish,   *
* distribute, sublicense, and/or sell copies of the Software, and to    *
* permit persons to whom the Software is furnished to do so, subject to *
* the following conditions:                                             *
*                                                                       *
* The above copyright notice and this permission notice shall be        *
* included in all copies or substantial portions of the Software.       *
*                                                                       *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  *
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  *
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     *
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                *
************************************************************************/

/************************************************************************
Changes:
************************************************************************/
#include <stdlib.h>
#include <string.h>

#include "lua.h"
#include "lauxlib.h"

#include "zlib.h"


/*
** =========================================================================
** compile time options wich determine available functionality
** =========================================================================
*/


/*
** =========================================================================
** zlib buffer metamethods
** =========================================================================
*/
#define ZBUFMETA        "zlib:buffer"

#define IDX_CB          1
#define IDX_CBUD        2
#define IDX_DICT        3
#define IDX_DICTUD      4

#define TINDEX          lua_upvalueindex(1)

#define STATE_ANY       -1
#define STATE_NONE      0
#define STATE_DEFLATE   1
#define STATE_INFLATE   2

typedef struct lzbuf lzbuf;

struct lzbuf
{
    int state;          /* indicate usable state */
    z_stream zstream;   /* zlib stream */
};

static lzbuf *lzbuf_new(lua_State *L)
{
    lzbuf *zbuf = (lzbuf*)lua_newuserdata(L, sizeof(lzbuf));

    luaL_getmetatable(L, ZBUFMETA);
    lua_setmetatable(L, -2);        /* set metatable */

    zbuf->state = STATE_NONE;

    zbuf->zstream.zalloc = Z_NULL;
    zbuf->zstream.zfree = Z_NULL;

    zbuf->zstream.next_out = Z_NULL;
    zbuf->zstream.avail_out = 0;
    zbuf->zstream.next_in = Z_NULL;
    zbuf->zstream.avail_in = 0;

    /* add an entry in upvalue table to store associated values */
    lua_pushlightuserdata(L, zbuf); /* key - lightuserdata */
    lua_newtable(L);                /* value - new table */
    lua_rawset(L, TINDEX);          /* add our table */

    return zbuf;
}

static void lzbuf_cleanup(lua_State *L, lzbuf *zbuf)
{
    if (zbuf && zbuf->state != STATE_NONE)
    {
        if (zbuf->state == STATE_DEFLATE)
            deflateEnd(&zbuf->zstream);
        if (zbuf->state == STATE_INFLATE)
            inflateEnd(&zbuf->zstream);
        zbuf->state = STATE_NONE;

        /* remove our table entry in upvale table */
        lua_pushlightuserdata(L, zbuf);
        lua_pushnil(L);
        lua_rawset(L, TINDEX);
    }
}

/* ====================================================================== */

static lzbuf *lzbuf_check(lua_State *L, int index, int state)
{
    lzbuf *zbuf = (lzbuf*)luaL_checkudata(L, index, ZBUFMETA);
    if (zbuf == NULL) luaL_argerror(L, index, "bad zlib stream");
    if ((state != STATE_ANY && zbuf->state != state) || zbuf->state == STATE_NONE)
        luaL_argerror(L, index, "attempt to use invalid zlib stream");
    return zbuf;
}

/* ====================================================================== */

static void lzbuf_pushtable(lua_State *L, void *key)
{
    lua_pushlightuserdata(L, key);
    lua_rawget(L, TINDEX);
}

/* ====================================================================== */

static void lzbuf_flush(lua_State *L, lzbuf *zbuf)
{
    int top = lua_gettop(L);

    lzbuf_pushtable(L, zbuf);

    lua_rawgeti(L, -1, IDX_CB);     /* function to call */
    lua_pushvalue(L, -3);           /* data to save */
    lua_rawgeti(L, -3, IDX_CBUD);   /* userdata */

    lua_pcall(L, 2, 0, 0);

    lua_settop(L, top-1);   /* remove used stuff and data from stack */
}

/* ====================================================================== */

static void lzbuf_get_dict(lua_State *L, lzbuf *zbuf)
{
    int top = lua_gettop(L);

    lzbuf_pushtable(L, zbuf);

    lua_rawgeti(L, -1, IDX_DICT);   /* function to call */
    lua_rawgeti(L, -2, IDX_DICTUD); /* userdata */

    lua_pcall(L, 1, 0, 0);

    lua_settop(L, top);     /* remove used stuff from stack */
}
/* ====================================================================== */

static int lzbuf_tostring(lua_State *L)
{
    char buf[100];
    lzbuf *zbuf = (lzbuf*)luaL_checkudata(L, 1, ZBUFMETA);
    if (zbuf == NULL) luaL_argerror(L, 1, "bad zlib stream");

    if (zbuf->state == STATE_NONE)
        strncpy(buf, "zlib stream (closed)", 100);
    else if (zbuf->state == STATE_DEFLATE)
        snprintf(buf, 100, "zlib deflate stream (%p)", lua_touserdata(L, 1));
    else if (zbuf->state == STATE_INFLATE)
        snprintf(buf, 100, "zlib inflate stream (%p)", lua_touserdata(L, 1));
    else
        snprintf(buf, 100, "%p", lua_touserdata(L, 1));

    lua_pushstring(L, buf);
    return 1;
}

static int lzbuf_gc(lua_State *L)
{
    lzbuf *zbuf = (lzbuf*)luaL_checkudata(L, 1, ZBUFMETA);
    if (zbuf == NULL) luaL_argerror(L, 1, "bad zlib stream");
    lzbuf_cleanup(L, zbuf);
    return 0;
}

/* ====================================================================== */

static int lzbuf_adler(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lua_pushnumber(L, zbuf->zstream.adler);
    return 1;
}

/* ====================================================================== */

static int lzbuf_data_type(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lua_pushnumber(L, zbuf->zstream.data_type);
    return 1;
}

/* ====================================================================== */

static int lzbuf_total_in(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lua_pushnumber(L, zbuf->zstream.total_in);
    return 1;
}

/* ====================================================================== */

static int lzbuf_total_out(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lua_pushnumber(L, zbuf->zstream.total_out);
    return 1;
}

/* ====================================================================== */

static int lzlib_deflate_init(lua_State *L)
{
    int level = luaL_optint(L, 1, Z_DEFAULT_COMPRESSION);
    int method = luaL_optint(L, 2, Z_DEFLATED);
    int windowBits = luaL_optint(L, 3, 15);
    int memLevel = luaL_optint(L, 4, 8);
    int strategy = luaL_optint(L, 5, Z_DEFAULT_STRATEGY);

    lzbuf *zbuf = lzbuf_new(L);

    int ret = deflateInit2(&zbuf->zstream, level, method, windowBits, memLevel, strategy);

    if (ret != Z_OK)
        lua_pushnil(L);
    else
        zbuf->state = STATE_DEFLATE;

    lua_pushnumber(L, ret);
    return 2;
}

static int lzlib_inflate_init(lua_State *L)
{
    int windowBits = luaL_optint(L, 1, 15);

    lzbuf *zbuf = lzbuf_new(L);

    int ret = inflateInit2(&zbuf->zstream, windowBits);

    if (ret != Z_OK)
        lua_pushnil(L);
    else
        zbuf->state = STATE_INFLATE;

    lua_pushnumber(L, ret);
    return 2;
}

/* ====================================================================== */

static int lzbuf_deflate(lua_State *L, lzbuf *zbuf)
{
    const char *next_in = luaL_checkstring(L, 2);
    int flush = luaL_optint(L, 3, Z_NO_FLUSH);
    int avail_in = lua_strlen(L, 2);
    int ret;
    luaL_Buffer b;
    luaL_buffinit(L, &b);

    zbuf->zstream.next_in = (char*)next_in;
    zbuf->zstream.avail_in = avail_in;

    for(;;)
    {
        zbuf->zstream.next_out = luaL_prepbuffer(&b);
        zbuf->zstream.avail_out = LUAL_BUFFERSIZE;

        /* munch some more */
        ret = deflate(&zbuf->zstream, flush);

        /* push gathered data */
        luaL_addsize(&b, LUAL_BUFFERSIZE - zbuf->zstream.avail_out);

        /* done processing? */
        if (ret == Z_STREAM_END || (ret == Z_OK && zbuf->zstream.avail_out > 0))
            break;

        /* error condition? */
        if (ret != Z_OK)
            break;
    }

    /* send gathered data if any */
    luaL_pushresult(&b);
    if (lua_strlen(L, -1) > 0)
        lzbuf_flush(L, zbuf);

    lua_pushnumber(L, ret);
    lua_pushnumber(L, avail_in - zbuf->zstream.avail_in);
    return 2;
}

static int lzbuf_inflate(lua_State *L, lzbuf *zbuf)
{
    const char *next_in = luaL_checkstring(L, 2);
    int flush = luaL_optint(L, 3, Z_NO_FLUSH);
    int avail_in = lua_strlen(L, 2);
    int ret;
    int dict = 0;
    luaL_Buffer b;
    luaL_buffinit(L, &b);

    zbuf->zstream.next_in = (char*)next_in;
    zbuf->zstream.avail_in = avail_in;

    for(;;)
    {
        zbuf->zstream.next_out = luaL_prepbuffer(&b);
        zbuf->zstream.avail_out = LUAL_BUFFERSIZE;

        /* bake some more */
        ret = inflate(&zbuf->zstream, flush);

        /* push gathered data */
        luaL_addsize(&b, LUAL_BUFFERSIZE - zbuf->zstream.avail_out);

        /* need dictionary? */
        if (ret == Z_NEED_DICT && !dict)
        {
            /* send gathered data before setting dictionary if any */
            luaL_pushresult(&b);
            if (lua_strlen(L, -1) > 0)
                lzbuf_flush(L, zbuf);
            else
                lua_pop(L, 1);

            /* get dictionary */
            lzbuf_get_dict(L, zbuf);
            dict = 1;

            /* reset buffer */
            luaL_buffinit(L, &b);
            continue;
        }

        /* done processing? */
        if (ret == Z_STREAM_END || (ret == Z_OK && zbuf->zstream.avail_out > 0))
            break;
        /* */
        if (ret == Z_BUF_ERROR && zbuf->zstream.avail_out > 0)
            break;
        /* error condition? */
        if (ret != Z_OK && ret != Z_BUF_ERROR)
            break;
    }
    /* send gathered data if any */
    luaL_pushresult(&b);
    if (lua_strlen(L, -1) > 0)
        lzbuf_flush(L, zbuf);

    lua_pushnumber(L, ret);
    lua_pushnumber(L, avail_in - zbuf->zstream.avail_in);
    return 2;
}

static int lzbuf_process(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);

    if (zbuf->state == STATE_DEFLATE)
        return lzbuf_deflate(L, zbuf);
    else if (zbuf->state == STATE_INFLATE)
        return lzbuf_inflate(L, zbuf);

    lua_pushliteral(L, "invalid stream state");
    lua_error(L);
    return 0;
}

/* ====================================================================== */

static int lzbuf_done(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lzbuf_cleanup(L, zbuf);
    return 0;
}

/* ====================================================================== */

static int lzbuf_callback(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    if (lua_gettop(L) < 2 || lua_isnil(L, 2))
    {
        lzbuf_pushtable(L, zbuf);
        lua_pushnil(L);
        lua_rawseti(L, -2, IDX_CB);     /* clear callback */
        lua_pushnil(L);
        lua_rawseti(L, -2, IDX_CBUD);   /* clear userdata */
    }
    else
    {
        luaL_checktype(L, 2, LUA_TFUNCTION);
        /* make sure we have an userdata field (even if nil) */
        lua_settop(L, 3);

        lzbuf_pushtable(L, zbuf);
        lua_pushvalue(L, 2);
        lua_rawseti(L, -2, IDX_CB);     /* save callback */
        lua_pushvalue(L, 3);
        lua_rawseti(L, -2, IDX_CBUD);   /* save callback user data */
    }
    return 0;
}

/* ====================================================================== */

static int lzbuf_dictionary(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_INFLATE);
    if (lua_gettop(L) < 2 || lua_isnil(L, 2))
    {
        lzbuf_pushtable(L, zbuf);
        lua_pushnil(L);
        lua_rawseti(L, -2, IDX_DICT);   /* clear callback */
        lua_pushnil(L);
        lua_rawseti(L, -2, IDX_DICTUD); /* clear userdata */
    }
    else
    {
        luaL_checktype(L, 2, LUA_TFUNCTION);
        /* make sure we have an userdata field (even if nil) */
        lua_settop(L, 3);

        lzbuf_pushtable(L, zbuf);
        lua_pushvalue(L, 2);
        lua_rawseti(L, -2, IDX_DICT);   /* save callback */
        lua_pushvalue(L, 3);
        lua_rawseti(L, -2, IDX_DICTUD); /* save callback user data */
    }
    return 0;
}

/* ====================================================================== */

static int lzbuf_set_dictionary(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    const char* dict = luaL_checkstring(L, 2);
    int dict_len = lua_strlen(L, 2);

    if (zbuf->state == STATE_DEFLATE)
        lua_pushnumber(L, deflateSetDictionary(&zbuf->zstream, dict, dict_len));
    else if (zbuf->state == STATE_INFLATE)
        lua_pushnumber(L, inflateSetDictionary(&zbuf->zstream, dict, dict_len));
    else
    {
        lua_pushliteral(L, "invalid zlib stream state");
        lua_error(L);
    }

    return 1;
}

/* ====================================================================== */

static int lzbuf_clone(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);
    lzbuf *zcopy = lzbuf_new(L);
    int ret = Z_OK;

    if (zbuf->state == STATE_DEFLATE)
        ret = deflateCopy(&zcopy->zstream, &zbuf->zstream);
    else if (zbuf->state == STATE_INFLATE)
        ret = inflateCopy(&zcopy->zstream, &zbuf->zstream);
    else
    {
        lua_pushliteral(L, "invalid zlib stream state");
        lua_error(L);
    }

    if (ret != Z_OK)
    {
        lua_pushnil(L);
        lua_pushnumber(L, ret);
        return 2;
    }

    /* copy callbacks */
    lzbuf_pushtable(L, zbuf);
    lzbuf_pushtable(L, zcopy);

    /* copy callback */
    lua_rawgeti(L, -2, IDX_CB);
    lua_rawseti(L, -2, IDX_CB);

    /* copy callback user data */
    lua_rawgeti(L, -2, IDX_CBUD);
    lua_rawseti(L, -2, IDX_CBUD);

    /* although this should not be needed at all.. copy it to be safe */
    if (zbuf->state == STATE_INFLATE)
    {
        /* copy dictionary callback */
        lua_rawgeti(L, -2, IDX_DICT);
        lua_rawseti(L, -2, IDX_DICT);

        /* copy dictionary callback user data */
        lua_rawgeti(L, -2, IDX_DICTUD);
        lua_rawseti(L, -2, IDX_DICTUD);
    }

    return 1;
}

/* ====================================================================== */

static int lzbuf_reset(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_ANY);

    if (zbuf->state == STATE_DEFLATE)
        lua_pushnumber(L, deflateReset(&zbuf->zstream));
    else if (zbuf->state == STATE_INFLATE)
        lua_pushnumber(L, inflateReset(&zbuf->zstream));
    else
    {
        lua_pushliteral(L, "invalid zlib stream state");
        lua_error(L);
    }

    return 1;
}

/* ====================================================================== */

static int lzbuf_params(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_DEFLATE);
    int level = luaL_checkint(L, 2);
    int strategy = luaL_checkint(L, 3);
    luaL_Buffer b;
    luaL_buffinit(L, &b);
    int ret;

    zbuf->zstream.next_in = NULL;
    zbuf->zstream.avail_in = 0;

    for(;;)
    {
        zbuf->zstream.next_out = luaL_prepbuffer(&b);
        zbuf->zstream.avail_out = LUAL_BUFFERSIZE;

        /* set params and munch as needed */
        ret = deflateParams(&zbuf->zstream, level, strategy);

        /* push gathered data */
        luaL_addsize(&b, LUAL_BUFFERSIZE - zbuf->zstream.avail_out);

        /* more output to come? */
        if (ret == Z_BUF_ERROR)
            continue;
        break;
    }

    /* send gathered data if any */
    luaL_pushresult(&b);
    if (lua_strlen(L, -1) > 0)
        lzbuf_flush(L, zbuf);

    lua_pushnumber(L, ret);
    return 1;
}

/* ====================================================================== */

static int lzbuf_prime(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_DEFLATE);
    int bits = luaL_checkint(L, 2);
    int value = luaL_checkint(L, 3);

    lua_pushnumber(L, deflatePrime(&zbuf->zstream, bits, value));
    return 1;
}

/* ====================================================================== */

static int lzbuf_sync(lua_State *L)
{
    lzbuf *zbuf = lzbuf_check(L, 1, STATE_INFLATE);
    const char *next_in = luaL_checkstring(L, 2);
    int avail_in = lua_strlen(L, 2);

    zbuf->zstream.next_in = (char*)next_in;
    zbuf->zstream.avail_in = avail_in;

    lua_pushnumber(L, inflateSync(&zbuf->zstream));
    lua_pushnumber(L, avail_in - zbuf->zstream.avail_in);
    return 1;
}


/*
** =========================================================================
** zlib functions
** =========================================================================
*/

static int lzlib_version(lua_State *L)
{
    lua_pushstring(L, zlibVersion());
    return 1;
}

/* ====================================================================== */
static int lzlib_adler32(lua_State *L)
{
    if (lua_gettop(L) == 0)
    {
        /* adler32 initial value */
        lua_pushnumber(L, adler32(0L, Z_NULL, 0));
    }
    else
    {
        /* update adler32 checksum */
        int adler = luaL_checkint(L, 1);
        const char* buf = luaL_checkstring(L, 2);
        int len = lua_strlen(L, 2);

        lua_pushnumber(L, adler32(adler, buf, len));
    }
    return 1;
}

/* ====================================================================== */
static int lzlib_crc32(lua_State *L)
{
    if (lua_gettop(L) == 0)
    {
        /* crc32 initial value */
        lua_pushnumber(L, crc32(0L, Z_NULL, 0));
    }
    else
    {
        /* update crc32 checksum */
        int crc = luaL_checkint(L, 1);
        const char* buf = luaL_checkstring(L, 2);
        int len = lua_strlen(L, 2);

        lua_pushnumber(L, crc32(crc, buf, len));
    }
    return 1;
}


/* ====================================================================== */

static int lzlib_compile_flags(lua_State *L)
{
    lua_pushnumber(L, zlibCompileFlags());
    return 1;
}

/* ====================================================================== */


static int lzlib_compress(lua_State *L)
{
    const char *next_in = luaL_checkstring(L, 1);
    int avail_in = lua_strlen(L, 1);
    int level = luaL_optint(L, 2, Z_DEFAULT_COMPRESSION);
    int method = luaL_optint(L, 3, Z_DEFLATED);
    int windowBits = luaL_optint(L, 4, 15);
    int memLevel = luaL_optint(L, 5, 8);
    int strategy = luaL_optint(L, 6, Z_DEFAULT_STRATEGY);

    int ret;
    luaL_Buffer b;
    luaL_buffinit(L, &b);

    z_stream zs;

    zs.zalloc = Z_NULL;
    zs.zfree = Z_NULL;

    zs.next_out = Z_NULL;
    zs.avail_out = 0;
    zs.next_in = Z_NULL;
    zs.avail_in = 0;

    ret = deflateInit2(&zs, level, method, windowBits, memLevel, strategy);

    if (ret != Z_OK)
    {
        lua_pushnil(L);
        lua_pushnumber(L, ret);
        return 2;
    }

    zs.next_in = (char*)next_in;
    zs.avail_in = avail_in;

    for(;;)
    {
        zs.next_out = luaL_prepbuffer(&b);
        zs.avail_out = LUAL_BUFFERSIZE;

        /* munch some more */
        ret = deflate(&zs, Z_FINISH);

        /* push gathered data */
        luaL_addsize(&b, LUAL_BUFFERSIZE - zs.avail_out);

        /* done processing? */
        if (ret == Z_STREAM_END)
            break;

        /* error condition? */
        if (ret != Z_OK)
            break;
    }

    /* cleanup */
    deflateEnd(&zs);

    luaL_pushresult(&b);
    lua_pushnumber(L, ret);
    return 2;
}

/* ====================================================================== */

static int lzlib_uncompress(lua_State *L)
{
    const char *next_in = luaL_checkstring(L, 1);
    int avail_in = lua_strlen(L, 1);
    int windowBits = luaL_optint(L, 2, 15);

    int ret;
    luaL_Buffer b;
    luaL_buffinit(L, &b);

    z_stream zs;

    zs.zalloc = Z_NULL;
    zs.zfree = Z_NULL;

    zs.next_out = Z_NULL;
    zs.avail_out = 0;
    zs.next_in = Z_NULL;
    zs.avail_in = 0;

    ret = inflateInit2(&zs, windowBits);

    if (ret != Z_OK)
    {
        lua_pushnil(L);
        lua_pushnumber(L, ret);
        return 2;
    }

    zs.next_in = (char*)next_in;
    zs.avail_in = avail_in;

    for(;;)
    {
        zs.next_out = luaL_prepbuffer(&b);
        zs.avail_out = LUAL_BUFFERSIZE;

        /* bake some more */
        ret = inflate(&zs, Z_FINISH);

        /* push gathered data */
        luaL_addsize(&b, LUAL_BUFFERSIZE - zs.avail_out);

        /* need dictionary? - no dictionary support here, so just quit */
        if (ret == Z_NEED_DICT)
            break;

        /* done processing? */
        if (ret == Z_STREAM_END)
            break;

        /* error condition? */
        if (ret != Z_BUF_ERROR)
            break;
    }

    /* cleanup */
    inflateEnd(&zs);

    luaL_pushresult(&b);
    lua_pushnumber(L, ret);
    return 2;
}


/* ====================================================================== */

static int lzlib_newindex(lua_State *L)
{
    lua_pushliteral(L, "attempt to change readonly table");
    lua_error(L);
    return 0;
}

/*
** =========================================================================
** Register functions
** =========================================================================
*/

#define ZC(s)   { #s, Z_ ## s },


static const struct
{
    const char* name;
    int value;
} zlib_constants[] =
{
    /* allowed flush values; see deflate() and inflate() for details */
    ZC(NO_FLUSH)        ZC(PARTIAL_FLUSH)   ZC(SYNC_FLUSH)
    ZC(FULL_FLUSH)      ZC(FINISH)          ZC(BLOCK)

    /* return codes for the compression/decompression functions */
    ZC(OK)              ZC(STREAM_END)      ZC(NEED_DICT)
    ZC(ERRNO)           ZC(STREAM_ERROR)    ZC(DATA_ERROR)
    ZC(MEM_ERROR)       ZC(BUF_ERROR)       ZC(VERSION_ERROR)

    /* compression levels */
    ZC(NO_COMPRESSION)  ZC(BEST_SPEED)      ZC(BEST_COMPRESSION)
    ZC(DEFAULT_COMPRESSION)

    /* compression strategy; see deflateInit2() for details */
    ZC(FILTERED)        ZC(HUFFMAN_ONLY)    ZC(RLE)
    ZC(DEFAULT_STRATEGY)

    /* possible values of the data_type field (though see inflate()) */
    ZC(BINARY)          ZC(ASCII)           ZC(UNKNOWN)

    /* The deflate compression method (the only one supported in this version) */
    ZC(DEFLATED)

    /* list terminator */
    { NULL, 0 }
};

/* ====================================================================== */

static const luaL_reg bufmm[] =
{
    {"callback",            lzbuf_callback          },
    {"dictionary",          lzbuf_dictionary        },

    {"set_dictionary",      lzbuf_set_dictionary    },
    {"params",              lzbuf_params            },
    {"prime",               lzbuf_prime             },
    {"sync",                lzbuf_sync              },

    {"reset",               lzbuf_reset             },
    {"process",             lzbuf_process           },
    {"done",                lzbuf_done              },

    {"clone",               lzbuf_clone             },

    {"adler",               lzbuf_adler             },
    {"data_type",           lzbuf_data_type         },
    {"total_in",            lzbuf_total_in          },
    {"total_out",           lzbuf_total_out         },

    {"__tostring",          lzbuf_tostring          },
    {"__gc",                lzbuf_gc                },
    {NULL, NULL}
};

static const luaL_reg zlib[] =
{
    {"version",             lzlib_version           },
    {"compile_flags",       lzlib_compile_flags     },
    {"adler32",             lzlib_adler32           },
    {"crc32",               lzlib_crc32             },

    {"deflate_init",        lzlib_deflate_init      },
    {"inflate_init",        lzlib_inflate_init      },

    {"compress",            lzlib_compress          },
    {"uncompress",          lzlib_uncompress        },

    {"__newindex",          lzlib_newindex          },

    {NULL, NULL}
};

LUALIB_API int luaopen_zlib(lua_State *L)
{
    int i = 0;  /* for contant registering */

    /* make sure header and library version are consistent */
    const char* version = zlibVersion();
    if (strcmp(version, ZLIB_VERSION))
    {
        lua_pushfstring(L, "zlib library version does not match - header: %s, library: %s", ZLIB_VERSION, version);
        lua_error(L);
    }

    /*
    ** create table to hold callback related information
    ** this table will be located at upvalue index 1 and will
    ** be shared by the metatable and the main sqlite table
    **
    ** each zlib stream handle will contain an entry in this table
    ** and the key will be the light user data 'version' of the
    ** zlib stream handle
    */
    lua_newtable(L);                    /* create table */

    /*
    ** Stack: utable
    */

    /* create new metatable for zlib compression structures */
    luaL_newmetatable(L, ZBUFMETA);
    lua_pushliteral(L, "__index");
    lua_pushvalue(L, -2);               /* push metatable */
    lua_rawset(L, -3);                  /* metatable.__index = metatable */

    /*
    ** Stack: utable, metatable
    */
    lua_pushvalue(L, -2);               /* push table for upvalue */
    luaL_openlib(L, NULL, bufmm, 1);

    lua_pop(L, 1);                      /* remove metatable from stack */

    /*
    ** Stack: utable
    */
    luaL_openlib(L, "zlib", zlib, 1);

    while (zlib_constants[i].name != NULL)
    {
        lua_pushstring(L, zlib_constants[i].name);
        lua_pushnumber(L, zlib_constants[i].value);
        lua_rawset(L, -3);
        ++i;
    }

    /* set zlib's metatable to itself - set as readonly (__newindex) */
    lua_pushvalue(L, -1);
    lua_setmetatable(L, -2);

    /*
    ** Stack: zlib table
    */
    return 1;
}
