/*
** $Id: lustring.c,v 1.1.1.1 2004/07/22 18:59:58 tngd Exp $
** Standard library for string operations and pattern-matching
** See Copyright Notice in lua.h
*/


#include <ctype.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wchar.h>
#include <wctype.h>
#include <locale.h>

/* utf8 library for use in (at least) WIN32 */
#if defined(__WIN32__)
#include <libutf8.h>
#endif

#include "lua.h"
#include "lauxlib.h"
#include "lualib.h"

typedef long sint32;	/* a signed version for size_t */

/* allocate space for unicode string (+1 for terminator)*/
#define udsize(size) (sizeof(wchar_t)*((size)+1)+sizeof(size_t))
#define wlen(wstr) *(((size_t*)wstr)-1)

static const char *umeta = "ustring";

#if defined(__WIN32__) && !defined(_LIBUTF8_H)

static int myswprintf(wchar_t* s, size_t maxlen, const wchar_t* format, ...) {
    int res;
    va_list args;
    va_start(args, format);
    res = vswprintf(s, format, args);
    va_end(args);
    return res;
}
#define swprintf myswprintf

#endif // __WIN32__

/***************************************************************************
    misc
***************************************************************************/
static sint32 posrelat(sint32 pos, size_t len) {
    /* relative string position: negative means back from end */
    return (pos>=0) ? pos : (sint32)len+pos+1;
}

/***************************************************************************
    userdata creation
***************************************************************************/

/* create ustring userdata - unicode string is passed by params */
static wchar_t *lustring_create(lua_State *L, size_t size) {
    size_t *us = lua_newuserdata(L, udsize(size));
    wchar_t *wstr = (wchar_t*)(us+1);
    wstr[size] = L'\0'; /* terminator! */
    *us = size;

    luaL_getmetatable(L, umeta); /* set metatable */
    lua_setmetatable(L, -2);
    return wstr;
}

/* create unicode string from multibyte character string */
static wchar_t *lustring_newc(lua_State *L, const char *str, size_t n) {
    size_t size = mbstowcs(NULL, str, 0);
    if (size != (size_t)-1) {
        wchar_t *wstr = lustring_create(L, size);
        mbstowcs(wstr, str, size);   /* convert string */
        return wstr;
    }
    else {
        luaL_error(L, "malformed multibyte character string");
    }
    return NULL;
}

/* create unicode string from multibyte string in the lua stack */
static wchar_t *lustring_newi(lua_State *L, int index) {
    size_t n;
    const char *str = luaL_checklstring(L, index, &n);
    return lustring_newc(L, str, n);
}

/* duplicate unicode string (used, for example, to create substrings) */
static wchar_t *lustring_dup(lua_State *L, const wchar_t *wstr, size_t n) {
    wchar_t *nwstr = lustring_create(L, n);
    wmemcpy(nwstr, wstr, n);
    return nwstr;
}

static wchar_t *lustring_check(lua_State *L, int index, size_t *l)
{
    size_t *us = (size_t*)luaL_checkudata(L, index, umeta);
    if (us == NULL) {
        /* check if we have a string, and if we do,
        ** convert and replace it */
        if (lua_type(L, index) != LUA_TSTRING) {
            luaL_typerror(L, index, "unicode string");
        }
        else {
            wchar_t *u = lustring_newi(L, index);
            us = ((size_t*)u)-1;
            lua_replace(L, index < 0 ? index - 1 : index);
        }
    }
    if (l) *l = *us;
    return (wchar_t*)(us+1);
}

static int lustring_valid(lua_State *L, int index) {
    return luaL_checkudata(L, index, umeta) != NULL ||
        lua_type(L, index) == LUA_TSTRING ||
        lua_type(L, index) == LUA_TNUMBER;
}

/***************************************************************************
    buffer handling
***************************************************************************/
typedef struct {
    lua_State *L;
    wchar_t buffer[LUAL_BUFFERSIZE];
    wchar_t *p;
} u_Buffer;

static void lustringB_buffinit(lua_State *L, u_Buffer *b) {
    b->L = L;
    b->p = b->buffer;
    lustring_dup(L, L"", 0);
}

static int lustringB_emptybuffer(u_Buffer *b) {
    lua_State *L = b->L;
    size_t n = b->p - b->buffer;
    size_t l;
    wchar_t *wstr = lustring_check(L, -1, &l);
    wchar_t *nwstr = lustring_create(L, l + n);
    wmemcpy(nwstr, wstr, l);
    wmemcpy(nwstr + l, b->buffer, n);
    lua_replace(L, -2);
    b->p = b->buffer;
    return 0;
}

#define lustringB_addwchar(b, wchar) \
    ((void)((b)->p < ((b)->buffer+LUAL_BUFFERSIZE) || lustringB_emptybuffer(b)), \
    (*(b)->p++ = (wchar_t)(wchar)))
/*  ((void)((B)->p < ((B)->buffer+LUAL_BUFFERSIZE) || luaL_prepbuffer(B)),
    (*(B)->p++ = (char)(c)))
*/

static void lustringB_addlwstr(u_Buffer *b, const wchar_t *wstr, size_t l) {
    while (l--)
        lustringB_addwchar(b, *wstr++);
}

#if 0
static void lustringB_addwstr(u_Buffer *b, const wchar_t *ws) {
    lustringB_addlwstr(b, ws, wcslen(ws));
}
#endif

static void lustringB_addvalue(u_Buffer *b) {
    size_t l;
    wchar_t *wstr = lustring_check(b->L, -1, &l);
    lustringB_addlwstr(b, wstr, l);
}

static void lustringB_pushresult(u_Buffer *b) {
    if (b->p != b->buffer)
        lustringB_emptybuffer(b);
}

/***************************************************************************
    userdata meta methods
***************************************************************************/

static int lustring_convert(lua_State *L) {
    if (lua_isnil(L, 1))
        return 0;
    lustring_newi(L, 1);
    return 1;
}

static int lustring_tostring(lua_State *L) {
#if 1 /*defined(__WIN32__) && !defined(_LIBUTF8_H)*/
    /* choose the algorithm that converts the string in one step,
    ** i believe that it can be better than using alternative below */
    size_t l;
    const wchar_t *wstr = lustring_check(L, 1, &l);
    size_t n = wcstombs(NULL, wstr, 0) + 1;
    char *str = (char*)lua_newuserdata(L, n+1);
    wcstombs(str, wstr, n);
    lua_pushlstring(L, str, n - 1);
    return 1;
#else
    mbstate_t state;
    size_t l;
    const wchar_t *wstr = lustring_check(L, 1, &l);
    const wchar_t *wstrend = wstr + l;
    luaL_Buffer b;
    luaL_buffinit(L, &b);

    memset(&state, 0, sizeof(mbstate_t));
    while (wstr && wstr < wstrend) {
        char *p = luaL_prepbuffer(&b);
        size_t w = wcsrtombs(p, &wstr, LUAL_BUFFERSIZE, &state);
        if (w == (size_t)-1) {
            luaL_pushresult(&b);
            luaL_error(L, "invalid unicode string");
        }
        luaL_addsize(&b, w);
    }
    luaL_pushresult(&b);
    return 1;
#endif
}

static int lustring_concat(lua_State *L) {
    size_t l1, l2;
    wchar_t *ws1 = lustring_check(L, 1, &l1);
    wchar_t *ws2 = lustring_check(L, 2, &l2);
    wchar_t *res = lustring_create(L, l1+l2);
    wmemcpy(res, ws1, l1);
    wmemcpy(res + l1, ws2, l2);
    return 1;
}

static int lustring_eq(lua_State *L) {
    size_t l1, l2;
    wchar_t *ws1 = lustring_check(L, 1, &l1);
    wchar_t *ws2 = lustring_check(L, 2, &l2);
    lua_pushboolean(L, l1 == l2 && wmemcmp(ws1, ws2, l1) == 0);
    return 1;
}

static int lustring_lt(lua_State *L) {
    size_t l1, l2;
    wchar_t *ws1 = lustring_check(L, 1, &l1);
    wchar_t *ws2 = lustring_check(L, 2, &l2);
    lua_pushboolean(L, wmemcmp(ws1, ws2, l1 > l2 ? l1 : l2) < 0);
    return 1;
}

static int lustring_le(lua_State *L) {
    size_t l1, l2;
    wchar_t *ws1 = lustring_check(L, 1, &l1);
    wchar_t *ws2 = lustring_check(L, 2, &l2);
    lua_pushboolean(L, wmemcmp(ws1, ws2, l1 > l2 ? l1 : l2) <= 0);
    return 1;
}

/***************************************************************************
    userdata methods
***************************************************************************/

static int lustring_len(lua_State *L) {
    size_t l;
    lustring_check(L, 1, &l);
    lua_pushnumber(L, (lua_Number)l);
    return 1;
}

static int lustring_sub(lua_State *L) {
    size_t l;
    wchar_t *wstr = lustring_check(L, 1, &l);
    sint32 start = posrelat(luaL_checklong(L, 2), l);
    sint32 end = posrelat(luaL_optlong(L, 3, -1), l);
    if (start < 1) start = 1;
    if (end > (sint32)l) end = (sint32)l;
    if (start <= end)
        lustring_dup(L, wstr + start - 1, end - start + 1);
    else
        lustring_newc(L, "", 0);
    return 1;
}

static int lustring_lower(lua_State *L) {
    size_t l;
    size_t i;
    wchar_t *wstr = lustring_check(L, 1, &l);
    wchar_t *lower = lustring_create(L, l);
    for (i = 0; i < l; ++i)
        lower[i] = towlower(wstr[i]);
    return 1;
}

static int lustring_upper(lua_State *L) {
    size_t l;
    size_t i;
    wchar_t *wstr = lustring_check(L, 1, &l);
    wchar_t *upper = lustring_create(L, l);
    for (i = 0; i < l; ++i)
        upper[i] = towupper(wstr[i]);
    return 1;
}

static int lustring_rep(lua_State *L) {
    size_t l;
    wchar_t *wstr = lustring_check(L, 1, &l);
    int n = luaL_checkint(L, 2);
    wchar_t *rep;
    if (n < 0) n = 0;
    rep = lustring_create(L, l*n);
    while (n-- > 0)
        wmemcpy(&rep[l*n], wstr, l);
    return 1;
}

static int lustring_byte(lua_State *L) {
    size_t l;
    wchar_t *wstr = lustring_check(L, 1, &l);
    sint32 pos = posrelat(luaL_optlong(L, 2, 1), l);
    if (pos <= 0 || (size_t)(pos) > l)  /* index out of range? */
        return 0;  /* no answer */
    lua_pushnumber(L, wstr[pos-1]);
    return 1;
}

static int lustring_char(lua_State *L) {
    int n = lua_gettop(L);  /* number of arguments */
    int i;
    wchar_t *wstr = lustring_create(L, n);
    for (i = 0; i < n; ++i)
        wstr[i] = luaL_checkint(L, i+1);
    return 1;
}

/*
** {======================================================
** PATTERN MATCHING
** =======================================================
*/

#ifndef MAX_CAPTURES
#define MAX_CAPTURES 32  /* arbitrary limit */
#endif


#define CAP_UNFINISHED	(-1)
#define CAP_POSITION	(-2)

typedef struct MatchState {
  const wchar_t *src_init;  /* init of source string */
  const wchar_t *src_end;  /* end (`\0') of source string */
  lua_State *L;
  int level;  /* total number of captures (finished or unfinished) */
  struct {
    const wchar_t *init;
    sint32 len;
  } capture[MAX_CAPTURES];
} MatchState;


#define ESC         L'%'
#define SPECIALS    L"^$*+?.([%-"


static wint_t check_capture(MatchState *ms, int l) {
  l -= '1';
  if (l < 0 || l >= ms->level || ms->capture[l].len == CAP_UNFINISHED)
    return luaL_error(ms->L, "invalid capture index");
  return l;
}


static int capture_to_close(MatchState *ms) {
  int level = ms->level;
  for (level--; level>=0; level--)
    if (ms->capture[level].len == CAP_UNFINISHED) return level;
  return luaL_error(ms->L, "invalid pattern capture");
}


static const wchar_t *luaI_classend(MatchState *ms, const wchar_t *p) {
  switch (*p++) {
    case ESC: {
      if (*p == '\0')
        luaL_error(ms->L, "malformed pattern (ends with `%')");
      return p+1;
    }
    case L'[': {
      if (*p == L'^') p++;
      do {  /* look for a `]' */
        if (*p == L'\0')
          luaL_error(ms->L, "malformed pattern (missing `]')");
        if (*(p++) == ESC && *p != L'\0')
          p++;  /* skip escapes (e.g. `%]') */
      } while (*p != L']');
      return p+1;
    }
    default: {
      return p;
    }
  }
}


static int match_class(wint_t c, wint_t cl) {
  int res;
  switch (towlower(cl)) {
    case L'a' : res = iswalpha(c); break;
    case L'c' : res = iswcntrl(c); break;
    case L'd' : res = iswdigit(c); break;
    case L'l' : res = iswlower(c); break;
    case L'p' : res = iswpunct(c); break;
    case L's' : res = iswspace(c); break;
    case L'u' : res = iswupper(c); break;
    case L'w' : res = iswalnum(c); break;
    case L'x' : res = iswxdigit(c); break;
    case L'z' : res = (c == 0); break;
    default: return (cl == c);
  }
  return (iswlower(cl) ? res : !res);
}


static int matchbracketclass(wint_t c, const wchar_t *p, const wchar_t *ec) {
  int sig = 1;
  if (*(p+1) == L'^') {
    sig = 0;
    p++;  /* skip the `^' */
  }
  while (++p < ec) {
    if (*p == ESC) {
      p++;
      if (match_class(c, *p))
        return sig;
    }
    else if ((*(p+1) == '-') && (p+2 < ec)) {
      p+=2;
      if ((wint_t)(*(p-2)) <= c && c <= (wint_t)(*p))
        return sig;
    }
    else if ((wint_t)(*p) == c) return sig;
  }
  return !sig;
}


static int luaI_singlematch(wint_t c, const wchar_t *p, const wchar_t *ep) {
  switch (*p) {
    case L'.': return 1;  /* matches any char */
    case ESC: return match_class(c, *(p+1));
    case L'[': return matchbracketclass(c, p, ep-1);
    default:  return ((wint_t)(*p) == c);
  }
}


static const wchar_t *match(MatchState *ms, const wchar_t *s, const wchar_t *p);


static const wchar_t *matchbalance(MatchState *ms, const wchar_t *s, const wchar_t *p) {
  if (*p == 0 || *(p+1) == 0)
    luaL_error(ms->L, "unbalanced pattern");
  if (*s != *p) return NULL;
  else {
    int b = *p;
    int e = *(p+1);
    int cont = 1;
    while (++s < ms->src_end) {
      if (*s == e) {
        if (--cont == 0) return s+1;
      }
      else if (*s == b) cont++;
    }
  }
  return NULL;  /* string ends out of balance */
}


static const wchar_t *max_expand(MatchState *ms, const wchar_t *s,
                                 const wchar_t *p, const wchar_t *ep) {
  sint32 i = 0;  /* counts maximum expand for item */
  while ((s+i)<ms->src_end && luaI_singlematch((wint_t)(*(s+i)), p, ep))
    i++;
  /* keeps trying to match with the maximum repetitions */
  while (i>=0) {
    const wchar_t *res = match(ms, (s+i), ep+1);
    if (res) return res;
    i--;  /* else didn't match; reduce 1 repetition to try again */
  }
  return NULL;
}


static const wchar_t *min_expand(MatchState *ms, const wchar_t *s,
                                 const wchar_t *p, const wchar_t *ep) {
  for (;;) {
    const wchar_t *res = match(ms, s, ep+1);
    if (res != NULL)
      return res;
    else if (s<ms->src_end && luaI_singlematch((wint_t)(*s), p, ep))
      s++;  /* try with one more repetition */
    else return NULL;
  }
}


static const wchar_t *start_capture(MatchState *ms, const wchar_t *s,
                                    const wchar_t *p, int what) {
  const wchar_t *res;
  int level = ms->level;
  if (level >= MAX_CAPTURES) luaL_error(ms->L, "too many captures");
  ms->capture[level].init = s;
  ms->capture[level].len = what;
  ms->level = level+1;
  if ((res=match(ms, s, p)) == NULL)  /* match failed? */
    ms->level--;  /* undo capture */
  return res;
}


static const wchar_t *end_capture(MatchState *ms, const wchar_t *s,
                                  const wchar_t *p) {
  int l = capture_to_close(ms);
  const wchar_t *res;
  ms->capture[l].len = s - ms->capture[l].init;  /* close capture */
  if ((res = match(ms, s, p)) == NULL)  /* match failed? */
    ms->capture[l].len = CAP_UNFINISHED;  /* undo capture */
  return res;
}


static const wchar_t *match_capture(MatchState *ms, const wchar_t *s, int l) {
  size_t len;
  l = check_capture(ms, l);
  len = ms->capture[l].len;
  if ((size_t)(ms->src_end-s) >= len &&
      memcmp(ms->capture[l].init, s, len) == 0)
    return s+len;
  else return NULL;
}


static const wchar_t *match(MatchState *ms, const wchar_t *s, const wchar_t *p) {
  init: /* using goto's to optimize tail recursion */
  switch (*p) {
    case L'(': {  /* start capture */
      if (*(p+1) == L')')  /* position capture? */
        return start_capture(ms, s, p+2, CAP_POSITION);
      else
        return start_capture(ms, s, p+1, CAP_UNFINISHED);
    }
    case L')': {  /* end capture */
      return end_capture(ms, s, p+1);
    }
    case ESC: {
      switch (*(p+1)) {
        case L'b': {  /* balanced string? */
          s = matchbalance(ms, s, p+2);
          if (s == NULL) return NULL;
          p+=4; goto init;  /* else return match(ms, s, p+4); */
        }
        case 'f': {  /* frontier? */
          const wchar_t *ep; wchar_t previous;
          p += 2;
          if (*p != L'[')
            luaL_error(ms->L, "missing `[' after `%%f' in pattern");
          ep = luaI_classend(ms, p);  /* points to what is next */
          previous = (s == ms->src_init) ? L'\0' : *(s-1);
          if (matchbracketclass((wint_t)(previous), p, ep-1) ||
             !matchbracketclass((wint_t)(*s), p, ep-1)) return NULL;
          p=ep; goto init;  /* else return match(ms, s, ep); */
        }
        default: {
          if (iswdigit((wint_t)(*(p+1)))) {  /* capture results (%0-%9)? */
            s = match_capture(ms, s, *(p+1));
            if (s == NULL) return NULL;
            p+=2; goto init;  /* else return match(ms, s, p+2) */
          }
          goto dflt;  /* case default */
        }
      }
    }
    case L'\0': {  /* end of pattern */
      return s;  /* match succeeded */
    }
    case L'$': {
      if (*(p+1) == L'\0')  /* is the `$' the last char in pattern? */
        return (s == ms->src_end) ? s : NULL;  /* check end of string */
      else goto dflt;
    }
    default: dflt: {  /* it is a pattern item */
      const wchar_t *ep = luaI_classend(ms, p);  /* points to what is next */
      int m = s<ms->src_end && luaI_singlematch((wint_t)(*s), p, ep);
      switch (*ep) {
        case L'?': {  /* optional */
          const wchar_t *res;
          if (m && ((res=match(ms, s+1, ep+1)) != NULL))
            return res;
          p=ep+1; goto init;  /* else return match(ms, s, ep+1); */
        }
        case L'*': {  /* 0 or more repetitions */
          return max_expand(ms, s, p, ep);
        }
        case L'+': {  /* 1 or more repetitions */
          return (m ? max_expand(ms, s+1, p, ep) : NULL);
        }
        case L'-': {  /* 0 or more repetitions (minimum) */
          return min_expand(ms, s, p, ep);
        }
        default: {
          if (!m) return NULL;
          s++; p=ep; goto init;  /* else return match(ms, s+1, ep); */
        }
      }
    }
  }
}



static const wchar_t *lmemfind(const wchar_t *s1, size_t l1,
                               const wchar_t *s2, size_t l2) {
  if (l2 == 0) return s1;  /* empty strings are everywhere */
  else if (l2 > l1) return NULL;  /* avoids a negative `l1' */
  else {
    const wchar_t *init;  /* to search for a `*s2' inside `s1' */
    l2--;  /* 1st char will be checked by `memchr' */
    l1 = l1-l2;  /* `s2' cannot be found after that */
    while (l1 > 0 && (init = (const wchar_t *)wmemchr(s1, *s2, l1)) != NULL) {
      init++;   /* 1st char is already checked */
      if (wmemcmp(init, s2+1, l2) == 0)
        return init-1;
      else {  /* correct `l1' and `s1' to try again */
        l1 -= init-s1;
        s1 = init;
      }
    }
    return NULL;  /* not found */
  }
}


static void push_onecapture(MatchState *ms, int i) {
  int l = ms->capture[i].len;
  if (l == CAP_UNFINISHED) luaL_error(ms->L, "unfinished capture");
  if (l == CAP_POSITION)
    lua_pushnumber(ms->L, (lua_Number)(ms->capture[i].init - ms->src_init + 1));
  else
    lustring_dup(ms->L, ms->capture[i].init, l);
}


static int push_captures(MatchState *ms, const wchar_t *s, const wchar_t *e) {
  int i;
  luaL_checkstack(ms->L, ms->level, "too many captures");
  if (ms->level == 0 && s) {  /* no explicit captures? */
    lustring_dup(ms->L, s, e-s);  /* return whole match */
    return 1;
  }
  else {  /* return all captures */
    for (i=0; i<ms->level; i++)
      push_onecapture(ms, i);
    return ms->level;  /* number of strings pushed */
  }
}


static int lustring_find(lua_State *L) {
  size_t l1, l2;
  const wchar_t *s = lustring_check(L, 1, &l1);
  const wchar_t *p = lustring_check(L, 2, &l2);
  sint32 init = posrelat(luaL_optlong(L, 3, 1), l1) - 1;
  if (init < 0) init = 0;
  else if ((size_t)(init) > l1) init = (sint32)l1;
  if (lua_toboolean(L, 4) ||  /* explicit request? */
      wcspbrk(p, SPECIALS) == NULL) {  /* or no special characters? */
    /* do a plain search */
    const wchar_t *s2 = lmemfind(s+init, l1-init, p, l2);
    if (s2) {
      lua_pushnumber(L, (lua_Number)(s2-s+1));
      lua_pushnumber(L, (lua_Number)(s2-s+l2));
      return 2;
    }
  }
  else {
    MatchState ms;
    int anchor = (*p == L'^') ? (p++, 1) : 0;
    const wchar_t *s1=s+init;
    ms.L = L;
    ms.src_init = s;
    ms.src_end = s+l1;
    do {
      const wchar_t *res;
      ms.level = 0;
      if ((res=match(&ms, s1, p)) != NULL) {
        lua_pushnumber(L, (lua_Number)(s1-s+1));  /* start */
        lua_pushnumber(L, (lua_Number)(res-s));   /* end */
        return push_captures(&ms, NULL, 0) + 2;
      }
    } while (s1++<ms.src_end && !anchor);
  }
  lua_pushnil(L);  /* not found */
  return 1;
}


static int gfind_aux(lua_State *L) {
  MatchState ms;
  size_t ls;
  const wchar_t *s = lustring_check(L, lua_upvalueindex(1), &ls);
  const wchar_t *p = lustring_check(L, lua_upvalueindex(2), NULL);
  const wchar_t *src;
  ms.L = L;
  ms.src_init = s;
  ms.src_end = s+ls;
  for (src = s + (size_t)lua_tonumber(L, lua_upvalueindex(3));
       src <= ms.src_end;
       src++) {
    const wchar_t *e;
    ms.level = 0;
    if ((e = match(&ms, src, p)) != NULL) {
      int newstart = e-s;
      if (e == src) newstart++;  /* empty match? go at least one position */
      lua_pushnumber(L, (lua_Number)newstart);
      lua_replace(L, lua_upvalueindex(3));
      return push_captures(&ms, src, e);
    }
  }
  return 0;  /* not found */
}


static int lustring_gfind(lua_State *L) {
    lustring_check(L, 1, NULL);
    lustring_check(L, 2, NULL);
    lua_settop(L, 2);
    lua_pushnumber(L, 0);
    lua_pushcclosure(L, gfind_aux, 3);
    return 1;
}

static void add_s(MatchState *ms, u_Buffer *b,
                   const wchar_t *s, const wchar_t *e) {
  lua_State *L = ms->L;
  if (lustring_valid(L, 3)) {
    size_t l;
    const wchar_t *news = lustring_check(L, 3, &l);
    size_t i;
    for (i=0; i<l; i++) {
      if (news[i] != ESC)
        lustringB_addwchar(b, news[i]);
      else {
        i++;  /* skip ESC */
        if (!iswdigit((wint_t)(news[i])))
          lustringB_addwchar(b, news[i]);
        else {
          int level = check_capture(ms, news[i]);
          push_onecapture(ms, level);
          lustringB_addvalue(b);  /* add capture to accumulated result */
        }
      }
    }
  }
  else {  /* is a function */
    int n;
    lua_pushvalue(L, 3);
    n = push_captures(ms, s, e);
    lua_call(L, n, 1);
    if (lustring_valid(L, -1))
      lustringB_addvalue(b);  /* add return to accumulated result */
    else
      lua_pop(L, 1);  /* function result is not a string: pop it */
  }
}


static int lustring_gsub(lua_State *L) {
  size_t srcl;
  const wchar_t *src = lustring_check(L, 1, &srcl);
  const wchar_t *p = lustring_check(L, 2, NULL);
  int max_s = luaL_optint(L, 4, srcl+1);
  int anchor = (*p == L'^') ? (p++, 1) : 0;
  int n = 0;
  MatchState ms;
  u_Buffer b;
  luaL_argcheck(L,
    lua_gettop(L) >= 3 && (lustring_valid(L, 3) || lua_isfunction(L, 3)),
    3, "string or function expected");
  lustringB_buffinit(L, &b);
  ms.L = L;
  ms.src_init = src;
  ms.src_end = src+srcl;
  while (n < max_s) {
    const wchar_t *e;
    ms.level = 0;
    e = match(&ms, src, p);
    if (e) {
      n++;
      add_s(&ms, &b, src, e);
    }
    if (e && e>src) /* non empty match? */
      src = e;  /* skip it */
    else if (src < ms.src_end)
      lustringB_addwchar(&b, *src++);
    else break;
    if (anchor) break;
  }
  lustringB_addlwstr(&b, src, ms.src_end-src);
  lustringB_pushresult(&b);
  lua_pushnumber(L, (lua_Number)n);  /* number of substitutions */
  return 2;
}

/* }====================================================== */

/* maximum size of each formatted item (> len(format('%99.99f', -1e308))) */
#define MAX_ITEM	512
/* maximum size of each format specification (such as '%-099.99d') */
#define MAX_FORMAT	20


static void luaI_addquoted(lua_State *L, u_Buffer *b, int arg) {
  size_t l;
  const wchar_t *s = lustring_check(L, arg, &l);
  lustringB_addwchar(b, L'"');
  while (l--) {
    switch (*s) {
      case L'"': case L'\\': case L'\n': {
        lustringB_addwchar(b, '\\');
        lustringB_addwchar(b, *s);
        break;
      }
      case L'\0': {
        lustringB_addlwstr(b, L"\\000", 4);
        break;
      }
      default: {
        lustringB_addwchar(b, *s);
        break;
      }
    }
    s++;
  }
  lustringB_addwchar(b, L'"');
}


static const wchar_t *scanformat(lua_State *L, const wchar_t *strfrmt,
                                 wchar_t *form, int *hasprecision) {
  const wchar_t *p = strfrmt;
  while (wcschr(L"-+ #0", *p)) p++;  /* skip flags */
  if (iswdigit((wint_t)*p)) p++;  /* skip width */
  if (iswdigit((wint_t)*p)) p++;  /* (2 digits at most) */
  if (*p == L'.') {
    p++;
    *hasprecision = 1;
    if (iswdigit((wint_t)(*p))) p++;  /* skip precision */
    if (iswdigit((wint_t)(*p))) p++;  /* (2 digits at most) */
  }
  if (iswdigit((wint_t)(*p)))
    luaL_error(L, "invalid format (width or precision too long)");
  if (p-strfrmt+2 > MAX_FORMAT)  /* +2 to include `%' and the specifier */
    luaL_error(L, "invalid format (too long)");
  form[0] = L'%';
  wcsncpy(form+1, strfrmt, p-strfrmt+1);
  form[p-strfrmt+2] = 0;
  return p;
}


static int lustring_format(lua_State *L) {
  int arg = 1;
  size_t sfl;
  const wchar_t *strfrmt = lustring_check(L, arg, &sfl);
  const wchar_t *strfrmt_end = strfrmt+sfl;
  u_Buffer b;
  lustringB_buffinit(L, &b);
  while (strfrmt < strfrmt_end) {
    if (*strfrmt != '%')
      lustringB_addwchar(&b, *strfrmt++);
    else if (*++strfrmt == '%')
      lustringB_addwchar(&b, *strfrmt++);  /* %% */
    else { /* format item */
      wchar_t form[MAX_FORMAT];  /* to store the format (`%...') */
      wchar_t buff[MAX_ITEM];  /* to store the formatted item */
      int hasprecision = 0;
      if (iswdigit((wint_t)(*strfrmt)) && *(strfrmt+1) == L'$')
        return luaL_error(L, "obsolete option (d$) to `format'");
      arg++;
      strfrmt = scanformat(L, strfrmt, form, &hasprecision);
      switch (*strfrmt++) {
        case L'c':  case L'd':  case L'i': {
          swprintf(buff, sizeof(buff), form, luaL_checkint(L, arg));
          break;
        }
        case L'o':  case L'u':  case L'x':  case L'X': {
          swprintf(buff, sizeof(buff), form, (unsigned int)(luaL_checknumber(L, arg)));
          break;
        }
        case L'e': case L'E': case L'f':
        case L'g': case L'G': {
          swprintf(buff, sizeof(buff), form, luaL_checknumber(L, arg));
          break;
        }
        case L'q': {
          luaI_addquoted(L, &b, arg);
          continue;  /* skip the `addsize' at the end */
        }
        case L's': {
          size_t l;
          const wchar_t *s = lustring_check(L, arg, &l);
          if (!hasprecision && l >= 100) {
            /* no precision and string is too long to be formatted;
               keep original string */
            lua_pushvalue(L, arg);
            lustringB_addvalue(&b);
            continue;  /* skip the `addsize' at the end */
          }
          else {
            swprintf(buff, sizeof(buff), form, s);
            break;
          }
        }
        default: {  /* also treat cases `pnLlh' */
          return luaL_error(L, "invalid option to `format'");
        }
      }
      lustringB_addlwstr(&b, buff, wcslen(buff));
    }
  }
  lustringB_pushresult(&b);
  return 1;
}

static int lustring_setlocale(lua_State *L) {
    static const int cat[] = {LC_ALL, LC_COLLATE, LC_CTYPE, LC_MONETARY,
                              LC_NUMERIC, LC_TIME};
    static const char *const catnames[] = {
        "all", "collate", "ctype", "monetary",
        "numeric", "time", NULL};
    const char *l = lua_tostring(L, 1);
    int op = luaL_findstring(luaL_optstring(L, 2, "all"), catnames);
    luaL_argcheck(L, l || lua_isnoneornil(L, 1), 1, "string expected");
    luaL_argcheck(L, op != -1, 2, "invalid option");
    lua_pushstring(L, setlocale(cat[op], l));
    return 1;
}

static const luaL_reg ustrlib[] = {
    {"__tostring", lustring_tostring},
    {"__concat", lustring_concat},
    {"__eq", lustring_eq},
    {"__lt", lustring_lt},
    {"__le", lustring_le},
#if 0
    {"__index", lustring_index},
    {"__newindex", lustring_newindex},
#endif

    {"setlocale", lustring_setlocale},
    {"new", lustring_convert},
    {"cstr", lustring_tostring},
    {"dup", lustring_tostring},

    {"len", lustring_len},
    {"sub", lustring_sub},
    {"lower", lustring_lower},
    {"upper", lustring_upper},
    {"char", lustring_char},
    {"rep", lustring_rep},
    {"byte", lustring_byte},
    {"format", lustring_format},
    {"find", lustring_find},
    {"gfind", lustring_gfind},
    {"gsub", lustring_gsub},
    {NULL, NULL}
};


/* open ustring library */
LUALIB_API int luaopen_ustring(lua_State *L) {
    /* create new metatable  */
    luaL_newmetatable(L, umeta);
    lua_pushliteral(L, "__index");
    lua_pushvalue(L, -2);               /* push metatable */
    lua_rawset(L, -3);                  /* metatable.__index = metatable */

    /* register (local) library metatable */
    luaL_openlib(L, NULL, ustrlib, 0);

    return 1;
}

LUALIB_API int luaopen_ustring_named(lua_State *L, const char *name) {
    /* create library environment */
    luaopen_ustring(L);
    /* global library table name */
    lua_pushstring(L, name);
    /* push library table */
    lua_pushvalue(L, -2);
    /* register it globaly */
    lua_settable(L, LUA_GLOBALSINDEX);
    return 1;
}
