/************************************************************************
* Author    : Tiago Dionizio (tngd@mega.ist.utl.pt)                     *
* Library   : lsqlite - a SQLite database binding for Lua 5             *
*                                                                       *
* Permission is hereby granted, free of charge, to any person obtaining *
* a copy of this software and associated documentation files (the       *
* "Software"), to deal in the Software without restriction, including   *
* without limitation the rights to use, copy, modify, merge, publish,   *
* distribute, sublicense, and/or sell copies of the Software, and to    *
* permit persons to whom the Software is furnished to do so, subject to *
* the following conditions:                                             *
*                                                                       *
* The above copyright notice and this permission notice shall be        *
* included in all copies or substantial portions of the Software.       *
*                                                                       *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  *
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  *
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     *
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                *
************************************************************************/

/************************************************************************
$Id: lsqlite.c,v 1.12 2004/08/31 18:35:37 tngd Exp $

To consider:
------------

EXPERIMENTAL APIS

* sqlite_progress_handler (implemented)
* sqlite_bind (implemented)
* sqlite_reset (implemented)

* sqlite_commit_hook
* sqlite_last_statement_changed

TODO?

* sqlite_set_authorizer

Not available in public code(?)

* sqlite_open_encrypted

Changes:
31-08-2004
----------
    * changed second return value of db:compile to be the rest of the
    sql statement that was not processed instead of the number of
    characters of sql not processed (situation in case of success).

09-04-2004
----------
    * renamed db:rows to db:urows
    * renamed db:prows to db:rows

    * added vm:get_unames()
    * added vm:get_utypes()
    * added vm:get_uvalues()

08-04-2004
----------
    * changed db:encoding() and db:version() to use sqlite_libencoding() and
    sqlite_libversion()

    * added vm:columns()
    * added vm:get_named_types()
    * added vm:get_named_values()

    * added db:prows - like db:rows but returns a table with the column values
    instead of returning multiple columns seperatly on each iteration

    * added compatibility functions idata,iname,itype,data,type

    * added luaopen_sqlite_module. allow the library to be loaded without
    setting a global variable. does the same as luaopen_sqlite, but does not
    set the global name "sqlite".

    * vm:bind now also returns an error string in case of error

31-03-2004 - 01-04-2004
-----------------------
    * changed most of the internals. now using references (luaL_ref) in
    most of the places

    * make the virtual machine interface seperate from the database
    handle. db:compile now returns a vm handle

    * added db:rows [for ... in db:rows(...) do ... end]

    * added db:close_vm

    * added sqlite.encode_binary and sqlite.decode_binary

    * attempt to do a strict checking on the return type of the user
    defined functions returned values

18-01-2004
----------
    * add check on sql function callback to ensure there is enough stack
    space to pass column values as parameters

03-12-2003
----------
    * callback functions now have to return boolean values to abort or
    continue operation instead of a zero or non-zero value

06-12-2003
----------
    * make version member of sqlite table a function instead of a string
************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "lua.h"
#include "lauxlib.h"

#include "sqlite.h"

/* compile time features */
#if !defined(SQLITE_OMIT_PROGRESS_CALLBACK)
    #define SQLITE_OMIT_PROGRESS_CALLBACK 0
#endif
#if !defined(SQLITE_OMIT_INMEMORYDB)
    #define SQLITE_OMIT_INMEMORYDB 0
#endif

typedef struct sdb sdb;
typedef struct sdb_vm sdb_vm;
typedef struct sdb_func sdb_func;

/* to use as C user data so i know what function sqlite is calling */
struct sdb_func
{
    /* references to associated lua values */
    int fn_step;
    int fn_finalize;
    int udata;

    sdb *db;
    int aggregate;
    int type;

    sdb_func *next;
};

/* information about database */
struct sdb
{
    /* associated lua state */
    lua_State *L;
    /* sqlite database handle */
    sqlite *db;

    /* sql functions stack usage */
    sdb_func *func;         /* top SQL function being called */

    /* references */
    int busy_cb;        /* busy callback */
    int busy_udata;

    int progress_cb;    /* progress handler */
    int progress_udata;

    int trace_cb;       /* trace callback */
    int trace_udata;

    int exec_cb;        /* exec callback */
    int exec_udata;
    int exec_columns;
};

/*
** =======================================================
** Database Operations
** =======================================================
*/

#define SQLITEMETA          ":sqlite"
#define SQLITEVMMETA        ":sqlite:vm"

/* virtual machine information */
struct sdb_vm
{
    sdb *db;            /* associated database handle */
    sqlite_vm *vm;      /* virtual machine */

    /* sqlite_step info */
    int columns;            /* number of columns in result */
    const char **values;    /* values of columns in result */
    const char **names;     /* names of columns followed by types of the columns */

    int temp;   /* temporary vm used in db:rows */
};

static sdb_vm *newvm(lua_State *L, sdb *db)
{
    sdb_vm *svm = (sdb_vm*)lua_newuserdata(L, sizeof(sdb_vm));

    luaL_getmetatable(L, SQLITEVMMETA);
    lua_setmetatable(L, -2);        /* set metatable */

    svm->db = db;
    svm->columns = 0;
    svm->values = NULL;
    svm->names = NULL;
    svm->vm = NULL;
    svm->temp = 0;

    /* add an entry on the database table */
    lua_pushlightuserdata(L, db);
    lua_rawget(L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata(L, svm);
    lua_pushvalue(L, -1);
    lua_rawset(L, -3);
    lua_pop(L, 1);

    return svm;
}

static int cleanupvm(lua_State *L, sdb_vm *svm)
{
    char *errmsg = NULL;
    int result;

    /* remove entry in database table - no harm if not present in the table */
    lua_pushlightuserdata(L, svm->db);
    lua_rawget(L, LUA_REGISTRYINDEX);
    lua_pushlightuserdata(L, svm);
    lua_pushnil(L);
    lua_rawset(L, -3);
    lua_pop(L, 1);

    svm->columns = 0;
    svm->values = NULL;
    svm->names = NULL;

    if (!svm->vm) return 0;

    result = sqlite_finalize(svm->vm, &errmsg);
    svm->vm = NULL;

    lua_pushnumber(L, result);
    if (result == SQLITE_OK)
        return 1;

    /* failure */
    lua_pushstring(L, errmsg);
    sqlite_freemem(errmsg);
    return 2;
}


static sdb_vm *lsqlite_getvm(lua_State *L, int index)
{
    sdb_vm *svm = (sdb_vm*)luaL_checkudata(L, index, SQLITEVMMETA);
    if (svm == NULL) luaL_argerror(L, index, "bad sqlite virtual machine");
    return svm;
}

static sdb_vm *lsqlite_checkvm(lua_State *L, int index)
{
    sdb_vm *svm = lsqlite_getvm(L, index);
    if (svm->vm == NULL) luaL_argerror(L, index, "attempt to use closed sqlite virtual machine");
    return svm;
}

static int dbvm_isopen(lua_State *L)
{
    sdb_vm *svm = lsqlite_getvm(L, 1);
    lua_pushboolean(L, svm->vm != NULL);
    return 1;
}

static int dbvm_tostring(lua_State *L)
{
    char buff[39];
    sdb_vm *svm = lsqlite_getvm(L, 1);
    if (svm->vm == NULL)
        strcpy(buff, "closed");
    else
        sprintf(buff, "%p", svm);
    lua_pushfstring(L, "sqlite virtual machine (%s)", buff);
    return 1;
}

static int dbvm_gc(lua_State *L)
{
    sdb_vm *svm = lsqlite_getvm(L, 1);
    if (svm->vm != NULL)  /* ignore closed vms */
        cleanupvm(L, svm);
    return 0;
}

/*
**
*/
static int dbvm_step(lua_State *L)
{
    int result;
    sdb_vm *svm = lsqlite_checkvm(L, 1);

    result = sqlite_step(svm->vm, &svm->columns, &svm->values, &svm->names);

    if (result != SQLITE_ROW)
        svm->values = NULL;

    if (result != SQLITE_ROW && result != SQLITE_DONE)
    {
        svm->columns = 0;
        svm->names = NULL;
    }

    lua_pushnumber(L, result);
    return 1;
}

static int dbvm_columns(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_pushnumber(L, svm->columns);
    return 1;
}

static int dbvm_get_values(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->values == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_newtable(L);
    for (n = 0; n < svm->columns;)
    {
        lua_pushstring(L, svm->values[n++]);
        lua_rawseti(L, -2, n);
    }

    lua_pushliteral(L, "n");
    lua_pushnumber(L, svm->columns);
    lua_rawset(L, -3);
    return 1;
}

static int dbvm_get_names(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_newtable(L);
    for (n = 0; n < svm->columns;)
    {
        lua_pushstring(L, svm->names[n++]);
        lua_rawseti(L, -2, n);
    }

    lua_pushliteral(L, "n");
    lua_pushnumber(L, svm->columns);
    lua_rawset(L, -3);
    return 1;
}

static int dbvm_get_types(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_newtable(L);
    for (n = 0; n < svm->columns;)
    {
        lua_pushstring(L, svm->names[svm->columns + n++]);
        lua_rawseti(L, -2, n);
    }

    lua_pushliteral(L, "n");
    lua_pushnumber(L, svm->columns);
    lua_rawset(L, -3);
    return 1;
}

static int dbvm_get_uvalues(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->values == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_checkstack(L, svm->columns);
    for (n = 0; n < svm->columns; ++n)
        lua_pushstring(L, svm->values[n]);

    return svm->columns;
}

static int dbvm_get_unames(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_checkstack(L, svm->columns);
    for (n = 0; n < svm->columns; ++n)
        lua_pushstring(L, svm->names[n]);

    return svm->columns;
}

static int dbvm_get_utypes(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_checkstack(L, svm->columns);
    for (n = 0; n < svm->columns; ++n)
        lua_pushstring(L, svm->names[svm->columns + n]);

    return svm->columns;
}

static int dbvm_get_named_values(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_newtable(L);
    for (n = 0; n < svm->columns; ++n)
    {
        lua_pushstring(L, svm->names[n]);
        lua_pushstring(L, svm->values[n]);
        lua_rawset(L, -3);
    }

    return 1;
}

static int dbvm_get_named_types(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int n;

    if (svm->names == NULL)
    {
        lua_pushliteral(L, "missuse of function");
        lua_error(L);
    }

    lua_newtable(L);
    for (n = 0; n < svm->columns; ++n)
    {
        lua_pushstring(L, svm->names[n]);
        lua_pushstring(L, svm->names[svm->columns + n]);
        lua_rawset(L, -3);
    }

    return 1;
}

/*
**
*/
static int dbvm_finalize(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    return cleanupvm(L, svm);
}

/*
** int sqlite_reset(sqlite_vm*, char **pzErrMsg);
*/
static int dbvm_reset(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    char *errmsg = NULL;
    int err = sqlite_reset(svm->vm, &errmsg);

    lua_pushnumber(L, err);

    if (errmsg == NULL)
        return 1;

    lua_pushstring(L, errmsg);
    sqlite_freemem(errmsg);
    return 2;
}

/*
** int sqlite_bind(sqlite_vm*, int idx, const char *value, int len, int copy);
*/
static int dbvm_bind(lua_State *L)
{
    sdb_vm *svm = lsqlite_checkvm(L, 1);
    int idx = luaL_checkint(L, 2);
    const char *value = NULL;
    int len = 0;
    int result;

    if (!lua_isnoneornil(L, 3))
    {
        value = luaL_checkstring(L, 3);
        len = lua_strlen(L, 3) + 1;
    }

    result = sqlite_bind(svm->vm, idx, value, len, 1);

    lua_pushnumber(L, result);
    if (result == SQLITE_OK)
        return 1;

    /* in case of error, return an error message too */
    lua_pushstring(L, sqlite_error_string(result));
    return 2;
}


/*
** When creating database handles, always creates a `closed' database handle
** before opening the actual database; so, if there is a memory error, the
** database is not left opened.
**
** Creates a new 'table' and leaves it in the stack
*/
static sdb *newdb (lua_State *L) {
    sdb *db = (sdb*)lua_newuserdata(L, sizeof(sdb));
    db->L = L;
    db->db = NULL;  /* database handle is currently `closed' */
    db->func = NULL;

    db->busy_cb =
    db->busy_udata =
    db->progress_cb =
    db->progress_udata =
    db->trace_cb =
    db->trace_udata =
    db->exec_cb =
    db->exec_udata =
    db->exec_columns = LUA_NOREF;

    luaL_getmetatable(L, SQLITEMETA);
    lua_setmetatable(L, -2);        /* set metatable */

    /* to keep track of 'open' virtual machines */
    lua_pushlightuserdata(L, db);
    lua_newtable(L);
    lua_rawset(L, LUA_REGISTRYINDEX);

    return db;
}

static void cleanupdb(lua_State *L, sdb *db)
{
    sdb_func *func;
    sdb_func *func_next;
    int top;

    /* free associated virtual machines */
    lua_pushlightuserdata(L, db);
    lua_rawget(L, LUA_REGISTRYINDEX);

    /* close all used handles */
    top = lua_gettop(L);
    lua_pushnil(L);
    while (lua_next(L, -2))
    {
        sdb_vm *svm = lua_touserdata(L, -1);
        cleanupvm(L, svm);

        lua_settop(L, top);
        lua_pushnil(L);
    }

    /* remove entry in lua registry table */
    lua_pushlightuserdata(L, db);
    lua_pushnil(L);
    lua_rawset(L, LUA_REGISTRYINDEX);

    /* 'free' all references */
    luaL_unref(L, LUA_REGISTRYINDEX, db->busy_cb);
    luaL_unref(L, LUA_REGISTRYINDEX, db->busy_udata);
    luaL_unref(L, LUA_REGISTRYINDEX, db->progress_cb);
    luaL_unref(L, LUA_REGISTRYINDEX, db->progress_udata);
    luaL_unref(L, LUA_REGISTRYINDEX, db->trace_cb);
    luaL_unref(L, LUA_REGISTRYINDEX, db->trace_udata);
    luaL_unref(L, LUA_REGISTRYINDEX, db->exec_cb);
    luaL_unref(L, LUA_REGISTRYINDEX, db->exec_udata);
    luaL_unref(L, LUA_REGISTRYINDEX, db->exec_columns);

    /* close database */
    sqlite_close(db->db);
    db->db = NULL;

    /* free associated memory with created functions */
    func = db->func;
    while (func)
    {
        func_next = func->next;
        luaL_unref(L, LUA_REGISTRYINDEX, func->fn_step);
        luaL_unref(L, LUA_REGISTRYINDEX, func->fn_finalize);
        luaL_unref(L, LUA_REGISTRYINDEX, func->udata);
        free(func);
        func = func_next;
    }
    db->func = NULL;
}

static sdb *lsqlite_getdb(lua_State *L, int index)
{
    sdb *db = (sdb*)luaL_checkudata(L, index, SQLITEMETA);
    if (db == NULL) luaL_argerror(L, index, "bad sqlite database");
    return db;
}

static sdb *lsqlite_checkdb(lua_State *L, int index)
{
    sdb *db = lsqlite_getdb(L, index);
    if (db->db == NULL) luaL_argerror(L, index, "attempt to use closed sqlite database");
    return db;
}

/*
** =======================================================
** Database Methods
** =======================================================
*/

static int db_isopen(lua_State *L)
{
    sdb *db = lsqlite_getdb(L, 1);
    lua_pushboolean(L, db->db != NULL ? 1 : 0);
    return 1;
}

static int db_last_insert_rowid(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    lua_pushnumber(L, sqlite_last_insert_rowid(db->db));
    return 1;
}

static int db_changes(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    lua_pushnumber(L, sqlite_changes(db->db));
    return 1;
}

static int db_interrupt(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    sqlite_interrupt(db->db);
    return 0;
}

/*
** Registering SQL functions:
*/

/*
** callback functions used when calling registered sql functions
*/
static void db_sql_normal_function(sqlite_func *context, int argc, const char **argv)
{
    sdb_func *func = (sdb_func*)sqlite_user_data(context);
    sdb *db = func->db;
    lua_State *L = db->L;
    int n;

    int top = lua_gettop(L);

    /* ensure there is enough space in the stack */
    lua_checkstack(L, argc + (func->aggregate ? 3 : 1));

    lua_rawgeti(L, LUA_REGISTRYINDEX, func->fn_step);   /* function to call */
    lua_rawgeti(L, LUA_REGISTRYINDEX, func->udata);     /* user data */

    if (func->aggregate)
    {
        void *p = sqlite_aggregate_context(context, 1); /* minimal mem usage */

        /* i think it is OK to use assume that using a light user data
        ** as an entry on LUA REGISTRY table will be unique */
        lua_pushlightuserdata(L, p);
        lua_rawget(L, LUA_REGISTRYINDEX);       /* context table */

        if (lua_isnil(L, -1))   /* not yet created? */
        {
            lua_pop(L, 1);
            lua_newtable(L);
            lua_pushlightuserdata(L, p);
            lua_pushvalue(L, -2);
            lua_rawset(L, LUA_REGISTRYINDEX);
        }

        lua_pushnumber(L, sqlite_aggregate_count(context)); /* push step count */
    }

    /* push params */
    for (n = 0; n < argc;)
        lua_pushstring(L, argv[n++]);

    if (func->aggregate)
    {
        lua_pcall(L, argc+3, 0, 0);
    }
    else if (!lua_pcall(L, argc+1, 1, 0))
    {
        if ((func->type == SQLITE_NUMERIC || func->type == SQLITE_ARGS) && lua_isnumber(L, -1))
            sqlite_set_result_double(context, lua_tonumber(L, -1));
        else if ((func->type == SQLITE_TEXT || func->type == SQLITE_ARGS) && lua_isstring(L, -1))
            sqlite_set_result_string(context, lua_tostring(L, -1), -1);
        else if (lua_isnil(L, -1))
            sqlite_set_result_string(context, NULL, 0);
        else
            sqlite_set_result_error(context, "invalid return value from sql function", -1);
    }
    else
    {
        sqlite_set_result_error(context, lua_tostring(L, -1), -1);
    }

    lua_settop(L, top);
}

static void db_sql_finalize_function(sqlite_func *context)
{
    sdb_func *func = (sdb_func*)sqlite_user_data(context);
    sdb *db = func->db;
    lua_State *L = db->L;
    void *p = sqlite_aggregate_context(context, 1); /* minimal mem usage */

    int top = lua_gettop(L);

    lua_rawgeti(L, LUA_REGISTRYINDEX, func->fn_finalize);   /* function to call */
    lua_rawgeti(L, LUA_REGISTRYINDEX, func->udata);         /* user data */

    lua_pushlightuserdata(L, p);
    lua_rawget(L, LUA_REGISTRYINDEX);   /* context table */

    if (lua_isnil(L, -1))       /* not yet created? shouldn't happen here... */
    {
        lua_pop(L, 1);
        lua_newtable(L);
        lua_pushlightuserdata(L, p);
        lua_pushvalue(L, -2);
        lua_rawset(L, LUA_REGISTRYINDEX);
    }

    lua_pushnumber(L, sqlite_aggregate_count(context)); /* push step count */

    if (!lua_pcall(L, 3, 1, 0))
    {
        if ((func->type == SQLITE_NUMERIC || func->type == SQLITE_ARGS) && lua_isnumber(L, -1))
            sqlite_set_result_double(context, lua_tonumber(L, -1));
        else if ((func->type == SQLITE_TEXT || func->type == SQLITE_ARGS) && lua_isstring(L, -1))
            sqlite_set_result_string(context, lua_tostring(L, -1), -1);
        else if (lua_isnil(L, -1))
            sqlite_set_result_string(context, NULL, 0);
        else
            sqlite_set_result_error(context, "[invalid return value from sql function]", -1);
    }
    else
    {
        sqlite_set_result_error(context, lua_tostring(L, -1), -1);
    }

    /* cleanup context */
    lua_pushlightuserdata(L, p);
    lua_pushnil(L);
    lua_rawset(L, LUA_REGISTRYINDEX);

    lua_settop(L, top);
}

/*
** Register a normal function
** Params: db, function name, number arguments, callback, datatype, user data
** Returns: true on sucess
**
** Normal function:
** Params: user data, params
** return: string or number
**
** Aggregate function:
** Params of step: user data, context, count, params
** Params of finalize: user data, context, count
** return of finalize: string or number
*/
static int db_register_function(lua_State *L, int aggregate)
{
    sdb *db = lsqlite_checkdb(L, 1);
    const char *name;
    int args;
    int type;
    int result;
    sdb_func *func;

    /* safety measure */
    if (aggregate) aggregate = 1;

    name = luaL_checkstring(L, 2);
    args = luaL_checkint(L, 3);
    luaL_checktype(L, 4, LUA_TFUNCTION);
    if (aggregate) luaL_checktype(L, 5, LUA_TFUNCTION);
    type = luaL_checkint(L, 5 + aggregate);

    /* validate datatype */
    switch (type)
    {
        case SQLITE_TEXT:
        case SQLITE_NUMERIC:
        case SQLITE_ARGS:
            break;
        default:
            /* default to SQLITE_ARGS since i cannot do a strict
            ** checking from lua... */
            type = SQLITE_ARGS;
            break;
    }

    func = (sdb_func*)malloc(sizeof(sdb_func));
    if (func == NULL)
    {
        lua_pushliteral(L, "out of memory");
        lua_error(L);
    }

    if (aggregate)
        result = !sqlite_create_aggregate(db->db, name, args, db_sql_normal_function, db_sql_finalize_function, func);
    else
        result = !sqlite_create_function(db->db, name, args, db_sql_normal_function, func);

    if (result)
    {
        sqlite_function_type(db->db, name, type);

        /* safety measures for userdata field to be present in the stack */
        lua_settop(L, 6 + aggregate);

        /* save registered function in db function list */
        func->db = db;
        func->aggregate = aggregate;
        func->type = type;
        func->next = db->func;
        db->func = func;

        /* save the setp/normal function callback */
        lua_pushvalue(L, 4);
        func->fn_step = luaL_ref(L, LUA_REGISTRYINDEX);
        /* save user data */
        lua_pushvalue(L, 6+aggregate);
        func->udata = luaL_ref(L, LUA_REGISTRYINDEX);

        if (aggregate)
        {
            lua_pushvalue(L, 5);
            func->fn_finalize = luaL_ref(L, LUA_REGISTRYINDEX);
        }
    }
    else
    {
        /* free allocated memory */
        free(func);
    }

    lua_pushboolean(L, result);
    return 1;
}

static int db_create_function(lua_State *L)
{
    return db_register_function(L, 0);
}

static int db_create_aggregate(lua_State *L)
{
    return db_register_function(L, 1);
}

/*
** trace callback:
** Params: database, callback function, userdata
**
** callback function:
** Params: userdata, sql
*/
static void db_trace_callback(void *user, const char *sql)
{
    sdb *db = (sdb*)user;
    lua_State *L = db->L;
    int top = lua_gettop(L);

    /* setup lua callback call */
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->trace_cb);    /* get callback */
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->trace_udata); /* get callback user data */
    lua_pushstring(L, sql); /* traced sql statement */

    /* call lua function */
    lua_pcall(L, 2, 0, 0);

    lua_settop(L, top);
}

static int db_trace(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);

    if (lua_gettop(L) < 2 || lua_isnil(L, 2))
    {
        luaL_unref(L, LUA_REGISTRYINDEX, db->trace_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->trace_udata);

        db->trace_cb = LUA_NOREF;
        db->trace_udata = LUA_NOREF;

        /* clear busy handler */
        sqlite_trace(db->db, NULL, NULL);
    }
    else
    {
        luaL_checktype(L, 2, LUA_TFUNCTION);

        /* make sure we have an userdata field (even if nil) */
        lua_settop(L, 3);

        luaL_unref(L, LUA_REGISTRYINDEX, db->trace_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->trace_udata);

        db->trace_udata = luaL_ref(L, LUA_REGISTRYINDEX);
        db->trace_cb = luaL_ref(L, LUA_REGISTRYINDEX);

        /* set busy handler */
        sqlite_trace(db->db, db_trace_callback, db);
    }

    return 0;
}


#if !defined(SQLITE_OMIT_PROGRESS_CALLBACK) || !SQLITE_OMIT_PROGRESS_CALLBACK

/*
** progress handler:
** Params: database, callback function, number of opcodes, userdata
**
** callback function:
** Params: userdata
** returns: 0 to return immediatly and return SQLITE_ABORT, non-zero to continue
*/
static int db_progress_callback(void *user)
{
    int result = 1; /* abort by default */
    sdb *db = (sdb*)user;
    lua_State *L = db->L;
    int top = lua_gettop(L);

    lua_rawgeti(L, LUA_REGISTRYINDEX, db->progress_cb);
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->progress_udata);

    /* call lua function */
    if (!lua_pcall(L, 1, 1, 0))
        result = lua_toboolean(L, -1);

    lua_settop(L, top);
    return result;
}

static int db_progress_handler(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);

    if (lua_gettop(L) < 2 || lua_isnil(L, 2))
    {
        luaL_unref(L, LUA_REGISTRYINDEX, db->progress_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->progress_udata);

        db->progress_cb = LUA_NOREF;
        db->progress_udata = LUA_NOREF;

        /* clear busy handler */
        sqlite_progress_handler(db->db, 0, NULL, NULL);
    }
    else
    {
        int n;
        luaL_checktype(L, 2, LUA_TFUNCTION);
        n = luaL_checkint(L, 3);

        /* make sure we have an userdata field (even if nil) */
        lua_settop(L, 4);

        luaL_unref(L, LUA_REGISTRYINDEX, db->progress_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->progress_udata);

        lua_pushvalue(L, 2);
        db->progress_cb = luaL_ref(L, LUA_REGISTRYINDEX);
        lua_pushvalue(L, 4);
        db->progress_udata = luaL_ref(L, LUA_REGISTRYINDEX);

        /* set progress callback */
        sqlite_progress_handler(db->db, n, db_progress_callback, db);
    }

    return 0;
}

#else /* #if !defined(SQLITE_OMIT_PROGRESS_CALLBACK) || !SQLITE_OMIT_PROGRESS_CALLBACK */

static int db_progress_handler(lua_State *L)
{
    lua_pushliteral(L, "progress callback support disabled at compile time");
    lua_error(L);
    return 0;
}

#endif /* #if !defined(SQLITE_OMIT_PROGRESS_CALLBACK) || !SQLITE_OMIT_PROGRESS_CALLBACK */

/*
** busy handler:
** Params: database, callback function, userdata
**
** callback function:
** Params: userdata, number of tries, table name
** returns: 0 to return immediatly and return SQLITE_BUSY, non-zero to try again
*/
static int db_busy_callback(void *user, const char *tname, int tries)
{
    int result = 0; /* abort by default */
    sdb *db = (sdb*)user;
    lua_State *L = db->L;
    int top = lua_gettop(L);

    lua_rawgeti(L, LUA_REGISTRYINDEX, db->busy_cb);
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->busy_udata);
    lua_pushnumber(L, tries);
    lua_pushstring(L, tname);

    /* call lua function */
    if (!lua_pcall(L, 3, 1, 0))
        result = lua_toboolean(L, -1);

    lua_settop(L, top);
    return result;
}

static int db_busy_handler(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);

    if (lua_gettop(L) < 2 || lua_isnil(L, 2))
    {
        luaL_unref(L, LUA_REGISTRYINDEX, db->busy_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->busy_udata);

        db->busy_cb = LUA_NOREF;
        db->busy_udata = LUA_NOREF;

        /* clear busy handler */
        sqlite_busy_handler(db->db, NULL, NULL);
    }
    else
    {
        luaL_checktype(L, 2, LUA_TFUNCTION);
        /* make sure we have an userdata field (even if nil) */
        lua_settop(L, 3);

        luaL_unref(L, LUA_REGISTRYINDEX, db->busy_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->busy_udata);

        db->busy_udata = luaL_ref(L, LUA_REGISTRYINDEX);
        db->busy_cb = luaL_ref(L, LUA_REGISTRYINDEX);

        /* set busy handler */
        sqlite_busy_handler(db->db, db_busy_callback, db);
    }

    return 0;
}

static int db_busy_timeout(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    int timeout = luaL_checkint(L, 2);
    sqlite_busy_timeout(db->db, timeout);
    return 0;
}

/*
** Params: db, sql, callback, user
** returns: code [, errmsg]
**
** Callback:
** Params: user, number of columns, data, names
** Returns: 0 to continue, other value will cause abort
*/
static int db_exec_callback(void* user, int columns, char **data, char **names)
{
    int result = 1; /* abort by default */
    sdb *db = (sdb*)user;
    lua_State *L = db->L;
    int n;

    int top = lua_gettop(L);

    /* function to call */
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->exec_cb);
    /* user data */
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->exec_udata);
    /* total number of rows in result */
    lua_pushnumber(L, columns);

    /* column values */
    lua_newtable(L);
    for (n = 0; n < columns;)
    {
        lua_pushstring(L, data[n++]);
        lua_rawseti(L, -2, n);
    }

    /* columns values */
    if (db->exec_columns == LUA_NOREF)
    {
        lua_newtable(L);
        for (n = 0; n < columns;)
        {
            lua_pushstring(L, names[n++]);
            lua_rawseti(L, -2, n);
        }
        db->exec_columns = luaL_ref(L, LUA_REGISTRYINDEX);
    }
    lua_rawgeti(L, LUA_REGISTRYINDEX, db->exec_columns);

    /* call lua function */
    if (!lua_pcall(L, 4, 1, 0))
        result = lua_toboolean(L, -1);

    lua_settop(L, top);

    return result;
}

static int db_exec(lua_State *L)
{
    int top = lua_gettop(L);
    sdb *db = lsqlite_checkdb(L, 1);
    const char *sql = luaL_checkstring(L, 2);
    char *errmsg = NULL;
    int result;

    if (top > 2 && !lua_isnil(L, 3))
    {
        luaL_checktype(L, 3, LUA_TFUNCTION);
        lua_settop(L, 4);       /* 'trap' userdata */

        db->exec_udata = luaL_ref(L, LUA_REGISTRYINDEX);
        db->exec_cb = luaL_ref(L, LUA_REGISTRYINDEX);
        db->exec_columns = LUA_NOREF;

        result = sqlite_exec(db->db, sql, db_exec_callback, db, &errmsg);

        luaL_unref(L, LUA_REGISTRYINDEX, db->exec_cb);
        luaL_unref(L, LUA_REGISTRYINDEX, db->exec_udata);
        luaL_unref(L, LUA_REGISTRYINDEX, db->exec_columns);
    }
    else
        result = sqlite_exec(db->db, sql, NULL, NULL, &errmsg);

    lua_pushnumber(L, result);

    if (errmsg != NULL)
    {
        lua_pushstring(L, errmsg);
        sqlite_freemem(errmsg);
        return 2;
    }

    return 1;
}

/*
** Params: db, sql
** returns: code, compiled length or error message
*/
static int db_compile(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    const char *sql = luaL_checkstring(L, 2);
    char *errmsg = NULL;
    const char *sqltail;
    int result;
    sdb_vm *svm = newvm(L, db);

    result = sqlite_compile(db->db, sql, &sqltail, &svm->vm, &errmsg);

    if (result != SQLITE_OK)
    {
        cleanupvm(L, svm);

        lua_pushnil(L);
        lua_pushstring(L, errmsg);
        sqlite_freemem(errmsg);
        lua_pushnumber(L, result);
        return 3;
    }

    /* vm already in the stack */
    lua_pushstring(L, sqltail);
    return 2;
}

static int db_do_next_row(lua_State *L, int packed)
{
    int result;
    sdb_vm *svm = lsqlite_checkvm(L, 1);

    result = sqlite_step(svm->vm, &svm->columns, &svm->values, &svm->names);

    if (result == SQLITE_ROW)
    {
        int i;

        if (packed)
        {
            lua_newtable(L);
            for (i = 0; i < svm->columns;)
            {
                lua_pushstring(L, svm->values[i]);
                lua_rawseti(L, -2, ++i);
            }
            lua_pushliteral(L, "n");
            lua_pushnumber(L, svm->columns);
            lua_rawset(L, -3);
            return 1;
        }
        else
        {
            lua_checkstack(L, svm->columns);
            for (i = 0; i < svm->columns; ++i)
                lua_pushstring(L, svm->values[i]);
            return svm->columns;
        }
    }

    if (result != SQLITE_DONE)
    {
        char *errmsg = NULL;
        result = sqlite_finalize(svm->vm, &errmsg);
        svm->vm = NULL;
        cleanupvm(L, svm);

        if (result != SQLITE_OK)
        {
            lua_pushstring(L, errmsg);
            sqlite_freemem(errmsg);
            lua_error(L);
        }
    }

    /* no error should generated here */
    cleanupvm(L, svm);
    return 0;
}

static int db_next_row(lua_State *L)
{
    return db_do_next_row(L, 0);
}

static int db_next_packed_row(lua_State *L)
{
    return db_do_next_row(L, 1);
}

static int db_do_rows(lua_State *L, int packed)
{
    sdb *db = lsqlite_checkdb(L, 1);
    const char *sql = luaL_checkstring(L, 2);
    int result;
    char *errmsg = NULL;
    sdb_vm *svm = newvm(L, db);
    svm->temp = 1;

    result = sqlite_compile(db->db, sql, NULL, &svm->vm, &errmsg);

    if (result != SQLITE_OK)
    {
        cleanupvm(L, svm);

        lua_pushstring(L, errmsg);
        sqlite_freemem(errmsg);
        lua_error(L);
    }

    lua_pushcfunction(L, packed ? db_next_packed_row : db_next_row);
    lua_pushvalue(L, -2);
    return 2;
}

static int db_rows(lua_State *L)
{
    return db_do_rows(L, 1);
}

/* packed version of db:rows */
static int db_urows(lua_State *L)
{
    return db_do_rows(L, 0);
}

static int db_tostring(lua_State *L)
{
    char buff[32];
    sdb *db = lsqlite_getdb(L, 1);
    if (db->db == NULL)
        strcpy(buff, "closed");
    else
        sprintf(buff, "%p", lua_touserdata(L, 1));
    lua_pushfstring(L, "sqlite database (%s)", buff);
    return 1;
}

static int db_close(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    cleanupdb(L, db);
    return 0;
}

static int db_close_vm(lua_State *L)
{
    sdb *db = lsqlite_checkdb(L, 1);
    /* cleanup temporary only tables? */
    int temp = lua_toboolean(L, 2);

    /* free associated virtual machines */
    lua_pushlightuserdata(L, db);
    lua_rawget(L, LUA_REGISTRYINDEX);

    /* close all used handles */
    lua_pushnil(L);
    while (lua_next(L, -2))
    {
        sdb_vm *svm = lua_touserdata(L, -1);

        if ((!temp || svm->temp) && svm->vm)
        {
            sqlite_finalize(svm->vm, NULL);
            svm->vm = NULL;
        }

        /* leave key in the stack */
        lua_pop(L, 1);
    }
    return 0;
}


static int db_gc(lua_State *L)
{
    sdb *db = lsqlite_getdb(L, 1);
    if (db->db != NULL)  /* ignore closed databases */
        cleanupdb(L, db);
    return 0;
}

/*
** =======================================================
** General library functions
** =======================================================
*/

static int lsqlite_version(lua_State *L)
{
    lua_pushstring(L, sqlite_libversion());
    return 1;
}

static int lsqlite_encoding(lua_State *L)
{
    lua_pushstring(L, sqlite_libencoding());
    return 1;
}

static int lsqlite_error_string(lua_State *L)
{
    int code = luaL_checkint(L, 1);
    lua_pushstring(L, sqlite_error_string(code));
    return 1;
}

static int lsqlite_complete(lua_State *L)
{
    const char* sql = luaL_checkstring(L, 1);
    lua_pushboolean(L, sqlite_complete(sql));
    return 1;
}

static int lsqlite_encode_binary(lua_State *L)
{
    const char *text = luaL_checkstring(L, 1);
    size_t len = lua_strlen(L, 1);

    char *etext = (char *)malloc(2 + (257 * len) / 254);
    int elen = sqlite_encode_binary((unsigned char*)text, len, (unsigned char*)etext);

    lua_pushlstring(L, etext, elen);
    free(etext);
    return 1;
}

static int lsqlite_decode_binary(lua_State *L)
{
    const char *text = luaL_checkstring(L, 1);
    size_t len = lua_strlen(L, 1);

    char *dtext = (char*)malloc(len);
    int dlen = sqlite_decode_binary((unsigned char*)text, (unsigned char*)dtext);

    if (dlen == -1)
    {
        free(dtext);
        lua_pushliteral(L, "malformed encoded string");
        lua_error(L);
    }

    lua_pushlstring(L, dtext, dlen);
    free(dtext);
    return 1;
}

static int lsqlite_open(lua_State *L)
{
    const char *filename = luaL_checkstring(L, 1);
    const int mode = luaL_optint(L, 2, 0);
    char *errmsg;

    sdb *db = newdb(L); /* create and leave in stack */

    db->db = sqlite_open(filename, mode, &errmsg);
    if (db->db != NULL)
        return 1;

    /* failed to open database */
    lua_pushnil(L);             /* push nil */
    lua_pushstring(L, errmsg);  /* push error message */
    sqlite_freemem(errmsg);
    return 2;
}

#if !defined(SQLITE_OMIT_INMEMORYDB) || !SQLITE_OMIT_INMEMORYDB

static int lsqlite_open_memory(lua_State *L)
{
    const int mode = luaL_optint(L, 1, 0);
    char *errmsg;

    sdb *db = newdb(L); /* create and leave in stack */

    db->db = sqlite_open(":memory:", mode, &errmsg);
    if (db->db != NULL)
        return 1;

    /* failed to open database */
    lua_pushnil(L);             /* push nil */
    lua_pushstring(L, errmsg);  /* push error message */
    sqlite_freemem(errmsg);
    return 2;
}

#else /* #if !defined(SQLITE_OMIT_INMEMORYDB) || !SQLITE_OMIT_INMEMORYDB */

static int lsqlite_open_memory(lua_State *L)
{
    lua_pushliteral(L, "in memory database support disabled at compile time");
    lua_error(L);
    return 0;
}

#endif

static int lsqlite_newindex(lua_State *L)
{
    lua_pushliteral(L, "attempt to change readonly table");
    lua_error(L);
    return 0;
}

/*
** =======================================================
** Register functions
** =======================================================
*/

#define SC(s)   { #s, SQLITE_ ## s },


static const struct
{
    const char* name;
    int value;
} sqlite_constants[] =
{
    /* error codes */
    SC(OK)          SC(ERROR)       SC(INTERNAL)    SC(PERM)
    SC(ABORT)       SC(BUSY)        SC(LOCKED)      SC(NOMEM)
    SC(READONLY)    SC(INTERRUPT)   SC(IOERR)       SC(CORRUPT)
    SC(NOTFOUND)    SC(FULL)        SC(CANTOPEN)    SC(PROTOCOL)
    SC(EMPTY)       SC(SCHEMA)      SC(TOOBIG)      SC(CONSTRAINT)
    SC(MISMATCH)    SC(MISUSE)      SC(NOLFS)       SC(AUTH)
    SC(FORMAT)      SC(NOTADB)

    /* sqlite_bind error code - not used at the moment */
    /*SC(RANGE)*/

    /* sqlite_step specific return values */
    SC(ROW)         SC(DONE)

    /* data types */
    SC(NUMERIC)     SC(TEXT)        SC(ARGS)

    /* terminator */
    { NULL, 0 }
};

/* ======================================================= */

static const luaL_reg dblib[] =
{
    {"isopen",              db_isopen               },
    {"last_insert_rowid",   db_last_insert_rowid    },
    {"changes",             db_changes              },
    {"interrupt",           db_interrupt            },

    {"create_function",     db_create_function      },
    {"create_aggregate",    db_create_aggregate     },

    {"trace",               db_trace                },
    {"progress_handler",    db_progress_handler     },
    {"busy_timeout",        db_busy_timeout         },
    {"busy_handler",        db_busy_handler         },

    {"compile",             db_compile              },
    {"rows",                db_rows                 },
    {"urows",               db_urows                },

    {"exec",                db_exec                 },
    {"close",               db_close                },
    {"close_vm",            db_close_vm             },

    {"__tostring",          db_tostring             },
    {"__gc",                db_gc                   },

    {NULL, NULL}
};

static const luaL_reg vmlib[] =
{
    {"isopen",              dbvm_isopen             },
    {"step",                dbvm_step               },
    {"reset",               dbvm_reset              },
    {"bind",                dbvm_bind               },
    {"finalize",            dbvm_finalize           },
    {"columns",             dbvm_columns            },
    {"get_values",          dbvm_get_values         },
    {"get_names",           dbvm_get_names          },
    {"get_types",           dbvm_get_types          },
    {"get_uvalues",         dbvm_get_uvalues        },
    {"get_unames",          dbvm_get_unames         },
    {"get_utypes",          dbvm_get_utypes         },

    {"get_named_values",    dbvm_get_named_values   },
    {"get_named_types",     dbvm_get_named_types    },

    {"__tostring",          dbvm_tostring           },
    {"__gc",                dbvm_gc                 },

    /* compatibility names (added by request) */
    {"idata",               dbvm_get_values         },
    {"inames",              dbvm_get_names          },
    {"itypes",              dbvm_get_types          },
    {"data",                dbvm_get_named_values   },
    {"type",                dbvm_get_named_types    },

    { NULL, NULL }
};

static const luaL_reg sqlitelib[] =
{
    {"version",         lsqlite_version         },
    {"encoding",        lsqlite_encoding        },
    {"error_string",    lsqlite_error_string    },
    {"complete",        lsqlite_complete        },
    {"encode_binary",   lsqlite_encode_binary   },
    {"decode_binary",   lsqlite_decode_binary   },
    {"open",            lsqlite_open            },
    {"open_memory",     lsqlite_open_memory     },
    {"__newindex",      lsqlite_newindex        },
    {NULL, NULL}
};

LUALIB_API int luaopen_sqlite_module(lua_State *L)
{
    int i = 0;

    /* create new metatable for database variables */
    luaL_newmetatable(L, SQLITEMETA);
    lua_pushliteral(L, "__index");
    lua_pushvalue(L, -2);               /* push metatable */
    lua_rawset(L, -3);                  /* metatable.__index = metatable */

    /* register sqlite database metatable */
    luaL_openlib(L, NULL, dblib, 0);

    /* remove metatable from stack */
    lua_pop(L, 1);


    /* create new metatable for virtual machine variables */
    luaL_newmetatable(L, SQLITEVMMETA);
    lua_pushliteral(L, "__index");
    lua_pushvalue(L, -2);               /* push metatable */
    lua_rawset(L, -3);                  /* metatable.__index = metatable */

    /* register sqlite database metatable */
    luaL_openlib(L, NULL, vmlib, 0);

    /* remove metatable from stack */
    lua_pop(L, 1);

    /* register (local) sqlite metatable */
    lua_newtable(L);
    luaL_openlib(L, NULL, sqlitelib, 0);

    /* add constants to global table */
    while (sqlite_constants[i].name != NULL)
    {
        lua_pushstring(L, sqlite_constants[i].name);
        lua_pushnumber(L, sqlite_constants[i].value);
        lua_rawset(L, -3);
        i++;
    }

    /* set sqlite's metatable to itself - set as readonly (__newindex) */
    lua_pushvalue(L, -1);
    lua_setmetatable(L, -2);

    return 1;
}

LUALIB_API int luaopen_sqlite(lua_State *L)
{
    /* global sqlite table name */
    lua_pushliteral(L, "sqlite");

    /* create sqlite environment */
    luaopen_sqlite_module(L);

    /* register it globaly */
    lua_settable(L, LUA_GLOBALSINDEX);

    return 1;
}
